// This component is taken from Twilight. If this functionality is moved to CoreUI then replace this.

import * as React from 'react';
import { Display, Layout, Position, Tooltip, TooltipProps } from 'twitch-core-ui';
import { v4 as uuid } from 'uuid';

export enum TestSelectors {
  BalloonInsideClickDetector = 'balloon-inside-click-detector',
  MouseEnterDetector = 'toggle-balloon-wrapper__mouse-enter-detector',
}

export interface PublicProps {
  /** specifies that the balloon's content should always exist in the DOM, regardless of whether or not it is visible.
   * needed for current channel-header resizing implementation.
   */
  alwaysMountBalloonContent?: boolean;
  display?: Display;
  /**
   * Automatically hides the balloon when a user clicks inside of it.
   */
  hideBalloonOnInsideClick?: boolean;
  /** specifies whether balloon starts in shown state when initially rendered. */
  openByDefault?: boolean;
  /** callback for when the balloon is toggled as a result of internal state change (not a manual toggle) */
  onToggle?: (isClosed: boolean) => void;
  tooltipProps?: TooltipProps;
  preventDefault?: boolean;
  stopPropagation?: boolean;
}

interface State {
  /** Prop passed to underlying Balloon to indicidate whether it should hide or show via CSS. */
  showBalloon: boolean;
  /** 'Sticky' state begins false; can only be set to true by mouseenter, click, or programmatic toggle via public toggleBallon method */
  hasInteracted: boolean;
}

type Props = PublicProps;

/**
 * Container for a click-to-toggle balloon that will close when
 * the user clicks outside. It should contain exactly two child elements:
 * - A button or link first, its onClick prop will be overwritten
 * - A core-ui Balloon element second, its show and toggle props will be over-written.
 * Doesn't mount the child Balloon unless alwaysMountBalloonContent prop is true, or user has
 * moused-over or clicked the corresponding Button.
 */
export class ToggleBalloonWrapper extends React.Component<Props, State> {
  public state = {
    showBalloon: !!this.props.openByDefault,
    hasInteracted: !!this.props.openByDefault,
  };
  private toggleBalloonId = uuid();

  public componentDidMount() {
    if (this.props.openByDefault) {
      document.addEventListener('click', this.handleGlobalClick, true);
    }
  }

  public componentWillUpdate(_nextProps: {}, nextState: State) {
    if (this.state.showBalloon !== nextState.showBalloon) {
      if (nextState.showBalloon) {
        document.addEventListener('click', this.handleGlobalClick, true);
      } else {
        document.removeEventListener('click', this.handleGlobalClick, true);
      }
    }
  }

  public componentWillUnmount() {
    document.removeEventListener('click', this.handleGlobalClick, true);
  }

  public render() {
    // tslint:disable-next-line:no-any
    const children = React.Children.toArray(this.props.children) as React.ReactElement<any>[];
    if (children.length !== 2) {
      throw new Error('ToggleBalloonWrapper should only be given two children: a clickable and a Balloon');
    }

    const [originalButton, originalBalloon] = children;

    const button = React.cloneElement(originalButton, {
      onClick: this.handleButtonClick,
    });

    let balloon: React.ReactElement<any> | null = null;

    if (this.state.hasInteracted || this.props.alwaysMountBalloonContent) {
      balloon = React.cloneElement(originalBalloon, {
        show: this.state.showBalloon,
      });

      if (this.props.hideBalloonOnInsideClick) {
        balloon = (
          <div data-test-selector={TestSelectors.BalloonInsideClickDetector} onClick={this.handleInsideBalloonClick}>
            {balloon}
          </div>
        );
      }
    }

    let buttonElement = button;

    if (this.props.tooltipProps) {
      buttonElement = (
        <Tooltip {...this.props.tooltipProps} >
          {button}
        </Tooltip>
      );
    }

    return (
      <Layout
        data-toggle-balloon-id={this.toggleBalloonId}
        display={this.props.display}
        position={Position.Relative}
      >
        <div style={{ display: 'inherit' }} onMouseEnter={this.handleMouseEnter} data-test-selector={TestSelectors.MouseEnterDetector}>
          {buttonElement}
        </div>
        {balloon}
      </Layout>
    );
  }

  /**
   * A public method for manually setting the state of the balloon from a parent component
   * This method does NOT call the onToggle callback and handling of the new state must be done by the parent
   */
  public toggleBalloon(showBalloon?: boolean) {
    if (showBalloon === undefined) {
      showBalloon = !this.state.showBalloon;
    }

    this.setState({
      showBalloon,
      hasInteracted: true,
    });
  }

  private handleButtonClick = (event: React.MouseEvent<HTMLElement>) => {
    if (this.props.preventDefault) {
      event.preventDefault();
    }

    if (this.props.stopPropagation) {
      event.stopPropagation();
    }

    this.setState((prevState) => ({
      showBalloon: !prevState.showBalloon,
      hasInteracted: true,
    }), () => {
      if (this.props.onToggle) {
        this.props.onToggle(!this.state.showBalloon);
      }
    });
  }

  private handleGlobalClick = (e: MouseEvent) => {
    const target = e.target as Element;
    if (!target.matches(`[data-toggle-balloon-id="${this.toggleBalloonId}"] *`)) {
      this.hideBalloon();
    }
  }

  private handleInsideBalloonClick = () => {
    this.hideBalloon();
  }

  private handleMouseEnter = () => {
    if (!this.state.hasInteracted) {
      this.setState({ hasInteracted: true });
    }
  }

  private hideBalloon() {
    this.setState({ showBalloon: false }, () => {
      if (this.props.onToggle) {
        this.props.onToggle(true);
      }
    });
  }
}
