import { ExtensionManifest } from './manifest';
import { SegmentMap, stopHosting, StopOptions, startBackend, startFrontend, hostFrontend } from '../../util/api';
import electron from '../../electron';
import { ExtensionAnchor, ExtensionMode, ExtensionPlatform, Configuration, ExtensionInstallationAbilities, ExtensionObject, ExtensionViewType } from 'extension-coordinator';
import { CertificateException } from './certificate-exception';
import React from 'react';
import { MobileOrientation } from '../../constants/mobile';

export const DeveloperRigUserId: string = '265737932';
export const ExtensionViewTypeKeys: { [key in ExtensionType]: ExtensionViewType; } = {
  [ExtensionAnchor.Component]: ExtensionViewType.Component,
  [ExtensionAnchor.Hidden]: ExtensionViewType.Hidden,
  [ExtensionAnchor.Overlay]: ExtensionViewType.VideoOverlay,
  [ExtensionAnchor.Panel]: ExtensionViewType.Panel,
  [ExtensionMode.Config]: ExtensionViewType.Config,
  [ExtensionMode.Dashboard]: ExtensionViewType.LiveConfig,
  [ExtensionPlatform.Mobile]: ExtensionViewType.Mobile,
};
const viewNames = [ExtensionViewType.Component, ExtensionViewType.Config, ExtensionViewType.LiveConfig, ExtensionViewType.Mobile, ExtensionViewType.Panel, ExtensionViewType.VideoOverlay];

export interface ExtensionFrameOptions {
  anchor: ExtensionAnchor;
  channelId: number;
  configuration?: Configuration;
  extension: ExtensionObject;
  features?: Partial<{
    bits: boolean,
  }>;
  iframeClassName: string;
  installationAbilities?: ExtensionInstallationAbilities;
  isPopout?: boolean;
  language?: string;
  locale?: string;
  loginId: number | null;
  mode: ExtensionMode;
  platform: ExtensionPlatform;
  trackingProperties: {};
}

export type ExtensionType = ExtensionAnchor |
  ExtensionMode.Config |
  ExtensionMode.Dashboard |
  ExtensionPlatform.Mobile;

export enum HostingResult {
  None = '',
  NotRunning = 'not running',
  Started = 'started',
  Running = 'running',
  Exited = 'exited',
}

export enum ViewerType {
  Broadcaster = 'Broadcaster',
  LoggedOut = 'Logged-Out Viewer',
  LoggedIn = 'Logged-In Viewer',
  Default = Broadcaster,
}

export interface RigProject {
  allowHttpBackend: boolean;
  backendCommand: string;
  backendFolderName: string;
  certificateExceptions: CertificateException[],
  extensionViews: RigExtensionView[],
  frontendCommand: string;
  frontendFolderName: string;
  manifest: ExtensionManifest;
  name: string;
  projectFolderPath: string;
  secret: string;
  usingRandomFrontendHostingPort: boolean;
  version: number;
  [key: string]: any;
}

export interface RigProjectReference {
  filePath: string;
  name: string;
  secret: string;
}

export interface RigExtensionView {
  configurationAnchor: ExtensionAnchor;
  displayChannelName: string;
  label: string;
  componentHeight?: number;
  orientation?: MobileOrientation;
  id: string;
  channelId: string;
  features: {
    isBitsEnabled: boolean;
    isChatEnabled: boolean;
    isSubscriptionStatusAvailable: boolean;
  };
  type: ExtensionType;
  viewerType: ViewerType;
  linked: boolean;
  linkedUserId: string;
  opaqueId?: string;
  isPopout: boolean;
  frameSize: FrameSize;
}

export interface FrameSize {
  height: number;
  width: number;
}

export interface ChannelSegments {
  [channelId: string]: SegmentMap;
}

export const RigContext = React.createContext<{
  userId: string;
  secret: string;
  isConfigurationHosted: boolean;
}>({
  isConfigurationHosted: false,
  secret: '',
  userId: '',
});

function getIsRunning(result: string) {
  return result === HostingResult.Started || result === HostingResult.Running;
}

export function getMode(type: ExtensionType): ExtensionMode {
  return type === ExtensionMode.Config ? ExtensionMode.Config :
    type === ExtensionMode.Dashboard ? ExtensionMode.Dashboard : ExtensionMode.Viewer;
}

export function getPort(rigProject: RigProject): number | undefined {
  const viewName = viewNames.filter((viewName: ExtensionViewType) => rigProject.manifest.views[viewName])[0];
  if (viewName) {
    const view = rigProject.manifest.views[viewName];
    if (view) {
      const url = new URL(view.viewerUrl);
      if (url.port) {
        const port = Number(url.port);
        if (!isNaN(port)) {
          return port;
        }
      }
    }
  }
}

export async function toggleBackend(rigProject: RigProject, backendState: string): Promise<string | undefined> {
  if (rigProject.backendCommand) {
    try {
      if (getIsRunning(backendState)) {
        const { backendResult } = await stopHosting(StopOptions.Backend);
        return backendResult;
      } else {
        const workingFolderPath = rigProject.backendFolderName ?
          electron.resolve(rigProject.projectFolderPath, rigProject.backendFolderName) :
          rigProject.projectFolderPath;
        await startBackend(rigProject.backendCommand, workingFolderPath);
        return HostingResult.Running;
      }
    } catch (ex) {
      return ex.message;
    }
  }
}

export async function toggleFrontend(rigProject: RigProject, frontendState: string): Promise<string | undefined> {
  if (rigProject.projectFolderPath || rigProject.frontendFolderName || rigProject.frontendCommand) {
    try {
      if (getIsRunning(frontendState)) {
        const { frontendResult } = await stopHosting(StopOptions.Frontend);
        return frontendResult;
      } else if (rigProject.frontendCommand) {
        await startFrontend(rigProject.frontendCommand, rigProject.frontendFolderName, rigProject.projectFolderPath);
        return HostingResult.Started;
      } else {
        const port = (!rigProject.usingRandomFrontendHostingPort && getPort(rigProject)) || 0;
        await hostFrontend(rigProject.frontendFolderName, port, rigProject.projectFolderPath);
        return HostingResult.Started;
      }
    } catch (ex) {
      return ex.message;
    }
  }
}

export function parseRigProject(fileText: string): RigProject {
  const project = JSON.parse(fileText) as RigProject;

  // Ensure the project has a version number.
  project.version = project.version || 1;

  // Replace the temporary new component value with the standard value since
  // there is no longer an old component value.
  for (const view of project.extensionViews) {
    if (view.type as string === 'NewComponent') {
      view.type = ExtensionAnchor.Component;
    }
  }

  if (project.version < 2) {
    // Replace "role" in extension views with "viewerType".
    for (const view of project.extensionViews) {
      view.viewerType = (view as any).role;
    }
    project.version = 2;
  }

  if (project.version < 3) {
    // Set "usingRandomFrontendHostingPort" to true.
    project.usingRandomFrontendHostingPort = true;
    project.version = 3;
  }

  return project;
}
