import { setupShallowTest } from '../tests/enzyme-util/shallow';
import { CreateProjectWizard, State } from './component';
import { timeout } from '../tests/utils';
import { Example } from '../util/api';
import { Modal } from 'twitch-core-ui';
import { CreateProjectWizardStepOne } from './components/step-one';
import { CodeGenerationOption } from './components/step-two';
import { NEW_EXTENSION_OPTION, ViewSupport } from './components/step-one';
import { RigProject } from '../core/models/rig';
import { ExtensionManifest } from '../core/models/manifest';
import { ExtensionState } from 'extension-coordinator';
import { Dialog } from '../dialog';
import { ExtensionClientsQueryResponse, CreateExtensionClientMutationResponse, ExtensionClient, SaveExtensionManifestMutationResponse, CurrentUserOrganizationsQueryResponse } from './queries/models';
import { VersionError } from './utils/validate-version';

const defaultExtensionManifest: ExtensionManifest = {
  authorName: 'author',
  bitsEnabled: false,
  configurationLocation: 'location',
  description: 'description',
  hasChatSupport: false,
  iconUrl: '',
  iconUrls: {
    '100x100': 'url',
  },
  id: 'extensionId',
  name: 'name',
  requestIdentityLink: false,
  sku: '',
  state: ExtensionState.Approved,
  summary: 'summary',
  vendorCode: '',
  version: '0.0.1',
  viewerSummary: 'viewer-summary',
  views: {
    panel: {
      canLinkExternalContent: false,
      height: 700,
      viewerUrl: 'url',
    },
  },
  whitelistedConfigUrls: [],
  whitelistedPanelUrls: [],
};

const defaultRigProject: RigProject = {
  allowHttpBackend: false,
  backendCommand: '',
  backendFolderName: '',
  certificateExceptions: [],
  extensionViews: [],
  frontendCommand: '',
  frontendFolderName: '',
  name: 'project name',
  manifest: defaultExtensionManifest,
  projectFolderPath: 'folderPath',
  secret: 'secret',
  version: 1,
};

const mockExtensionClient: ExtensionClient = {
  __typename: 'ExtensionClient',
  id: 'extensionId',
  createdAt: 'createdAt',
  name: 'extension name',
};

const mockExtensionClientsResponse: ExtensionClientsQueryResponse = {
  extensionClients: {
    __typename: 'ExtensionClientConnection',
    edges: [{
      __typename: 'ExtensionClientEdge',
      node: mockExtensionClient,
    }],
  },
};

const mockCreateExtensionClientMutationResponse: CreateExtensionClientMutationResponse = {
  createExtensionClient: {
    __typename: 'CreateExtensionClientPayload',
    client: mockExtensionClient,
    error: null,
  },
};

const mockCreateExtensionManifestMutationResponse: SaveExtensionManifestMutationResponse = {
  saveExtensionManifest: {
    __typename: 'SaveExtensionManifestPayload',
    error: null,
    manifest: {
      __typename: 'ExtensionManifest',
      id: 'extensionId',
      version: '0.0.1',
    }
  }
};

const mockCurrentUserOrganizationsResponse = jest.fn();

const mockExtensionGuide: Example = {
  backendCommand: 'backendCommand',
  backendFolderName: 'backendFolderName',
  description: 'description',
  expectedDuration: 11,
  frontendCommand: '',
  frontendFolderName: 'frontendFolderName',
  id: 0,
  isGuide: false,
  npm: ['npm'],
  repository: 'repository-owner/repository-name',
  title: 'title',
};

function mockApiFunctions() {
  return {
    ...require.requireActual('../util/api'),
    createProject: jest.fn().mockImplementation(() => Promise.resolve('filePath')),
    fetchExamples: jest.fn().mockImplementation(() => Promise.resolve([mockExtensionGuide])),
    fetchExtensionManifest: jest.fn().mockImplementation((id, _version) => id === 'error' ? Promise.reject(new Error('error')) : Promise.resolve({ id })),
    fetchExtensionSecret: jest.fn().mockResolvedValue('secret'),
    showOpenDialog: jest.fn().mockImplementation((_options) => Promise.resolve('folderPath')),
  };
}
jest.mock('../util/api', () => mockApiFunctions());
const api = require.requireMock('../util/api');
jest.mock('../util/graphql-client', () => ({
  graphqlClient: {
    request: jest.fn().mockImplementation((_, variables) => {
      if (!variables) {
        return mockCurrentUserOrganizationsResponse();
      } else if (variables.extensionName) {
        return mockCreateExtensionClientMutationResponse;
      } else if (variables.input) {
        return mockCreateExtensionManifestMutationResponse;
      } else if (variables.hasOwnProperty('organizationID')) {
        return mockExtensionClientsResponse;
      }
    }),
  },
}));

describe('CreateProjectWizard', () => {
  const setupShallow = setupShallowTest(CreateProjectWizard, () => ({
    userId: 'userId',
    closeHandler: jest.fn(),
    saveHandler: jest.fn()
  }));

  it('invokes closeHandler when cancel button is clicked', () => {
    const { props, wrapper } = setupShallow();
    const modal = wrapper.find(Dialog).dive().find(Modal);
    modal.prop('footerProps').secondaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
    expect(props.closeHandler).toHaveBeenCalled();
  });

  it('renders only personal extensions when the current user doesn\'t belong to any org', async () => {
    const mockOrgResp: CurrentUserOrganizationsQueryResponse = {
        currentUser:  {
        __typename: "User",
        id: "current_user_id",
        organizations: null,
      },
    };
    mockCurrentUserOrganizationsResponse.mockReturnValueOnce(
      mockOrgResp,
    );
    const { wrapper } = setupShallow();
    await wrapper.find(CreateProjectWizardStepOne).props().onRefreshExtensionsClick();
    wrapper.update();
    expect(wrapper.find(CreateProjectWizardStepOne).props().extensions.length).toBe(1);
  });

  it('renders only personal extensions when the current user doesn\'t belong to any org', async () => {
    const mockOrgResp: CurrentUserOrganizationsQueryResponse = {
      currentUser:  {
        __typename: "User",
        id: "current_user_id",
        organizations:  [{
          __typename: "Organization",
          id: "test_org_id",
        }],
      },
    };
    mockCurrentUserOrganizationsResponse.mockReturnValueOnce(
      mockOrgResp,
    );
    const { wrapper } = setupShallow();
    await wrapper.find(CreateProjectWizardStepOne).props().onRefreshExtensionsClick();
    wrapper.update();
    expect(wrapper.find(CreateProjectWizardStepOne).props().extensions.length).toBe(2);
  });

  describe('when moving from step 1 to step 2', () => {
    it('moves ahead with valid data', async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 1,
        extensionId: 'extensionId',
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectFolderPath: 'folderPath',
        projectName: 'project name',
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(2);
    });

    it("doesn't move ahead without an extension id", async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 1,
        extensionId: '',
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectFolderPath: 'folderPath',
        projectName: 'project name',
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(1);
    });

    it("creates an extension if extension id is set to new extension", async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 1,
        extensionId: NEW_EXTENSION_OPTION,
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: true,
        mustSelectProjectFolderPath: false,
        projectFolderPath: 'folderPath',
        projectName: 'project name',
        supportedViews: new Set([ViewSupport.Panel]),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('extensionId')).toBe('extensionId');
    });

    it("doesn't move ahead if there is an error for extension version", async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 1,
        extensionId: 'extensionId',
        extensions: [],
        extensionVersion: '',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectFolderPath: 'folderPath',
        projectName: 'project name',
        supportedViews: new Set(),
        versionError: VersionError.Empty,
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(1);
    });

    it("doesn't move ahead without a project name", async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 1,
        extensionId: 'extensionId',
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectFolderPath: 'folderPath',
        projectName: '',
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(1);
    });

    it("doesn't move ahead without a supported view type if creating a new extension", async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 1,
        extensionId: NEW_EXTENSION_OPTION,
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectFolderPath: 'folderPath',
        projectName: 'project name',
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(1);
    });
  })


  describe('when moving from step 2 to step 3', () => {
    it('moves ahead if path is provided and code generation option is none', async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 2,
        extensionId: 'extensionId',
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectName: 'project name',
        projectFolderPath: 'folderPath',
        codeGeneration: CodeGenerationOption.None,
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(3);
    });

    it('moves ahead if code generation option is guide and a guide is selected', async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 2,
        extensionId: 'extensionId',
        extensionVersion: '0.0.1',
        extensions: [],
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectName: 'project name',
        projectFolderPath: 'folderPath',
        codeGeneration: CodeGenerationOption.Guide,
        extensionGuide: mockExtensionGuide,
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(3);
    });

    it("doesn't move ahead if code generation option is guide but no guide is selected", async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 2,
        extensionId: 'extensionId',
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectName: 'project name',
        projectFolderPath: 'folderPath',
        codeGeneration: CodeGenerationOption.Guide,
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(2);
    });

    it("doesn't move ahead without a project folder path", async () => {
      const { wrapper } = setupShallow();
      const state: State = {
        currentStep: 2,
        extensionId: 'extensionId',
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectName: 'project name',
        projectFolderPath: '',
        codeGeneration: CodeGenerationOption.None,
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(wrapper.state('currentStep')).toBe(2);
    });
  });

  describe('when completing step 3', () => {
    it('calls the saveHandler', async () => {
      const { props, wrapper } = setupShallow();
      const state: State = {
        currentStep: 3,
        extensionId: 'extensionId',
        extensions: [],
        extensionVersion: '0.0.1',
        mustSelectCodeOption: false,
        mustSelectExtension: false,
        mustSelectProjectFolderPath: false,
        projectName: 'project name',
        projectFolderPath: 'folderPath',
        projectFilePath: 'filePath',
        project: defaultRigProject,
        codeGeneration: CodeGenerationOption.None,
        supportedViews: new Set(),
      };
      wrapper.setState(state);
      const modal = wrapper.find(Dialog).dive().find(Modal);
      modal.prop('footerProps').primaryButtonProps!.onClick!({} as React.MouseEvent<HTMLElement>);
      await timeout();
      expect(props.saveHandler).toHaveBeenCalledWith(
        defaultRigProject,
        'filePath',
        'secret',
      );
    });
  });
});
