import * as React from 'react';
import { RigProject } from '../core/models/rig';
import { Layout } from 'twitch-core-ui';
import { Dialog } from '../dialog';
import { CreateProjectWizardStepOne, NEW_EXTENSION_OPTION, ViewSupport } from './components/step-one';
import { CreateProjectWizardStepTwo, CodeGenerationOption } from './components/step-two';
import { CreateProjectWizardStepThree, CreateProjectStatus } from './components/step-three';
import { graphqlClient } from '../util/graphql-client';
import { extensionClientsQuery } from './queries/extension-clients';
import { createExtensionClientMutation } from './queries/create-extension-client';
import { createExtensionManifestMutation } from './queries/create-extension-manifest';
import { currentUserOrganizationsQuery } from './queries/user-organizations';
import { Example, createProject, fetchExtensionSecret, fetchExamples, fetchExtensionManifest } from '../util/api';
import { ExtensionManifest } from '../core/models/manifest';
import { validateVersion, VersionError } from './utils/validate-version';
import { CurrentUserOrganizationsQueryResponse, ExtensionClientsQueryResponse, CreateExtensionClientMutationResponse, CreateExtensionClientMutationVariables, buildMutationInputForCreateExtensionManifest, SaveExtensionManifestMutationResponse, CreateExtensionClientError } from './queries/models';
import { LocalStorage } from '../util/local-storage';
import { ProjectNameError, validateProjectName, InputError } from './utils/validate-project-name';
import './component.sass';
import electron from '../electron';
import { timeout } from '../tests/utils';

const extensionNameRegex = /Extension (.+) \d+/;
const finalStepNumber = 3;
const initialExtensionVersion = '0.0.1';

interface PublicProps {
  userId: string;
  closeHandler: () => void;
  saveHandler: (project: RigProject, filePath: string, secret: string) => Promise<void>;
}

interface Extension {
  id: string;
  name: string;
}

export interface State {
  codeGeneration?: CodeGenerationOption;
  createProjectStatus?: CreateProjectStatus;
  currentStep: number;
  extensionGuide?: Example;
  extensionId?: string;
  extensionManifest?: ExtensionManifest;
  extensions: Extension[];
  extensionSecret?: string;
  extensionVersion: string;
  isLoadingExtensions?: boolean;
  isLoadingNextStep?: boolean;
  mustSelectCodeOption: boolean;
  mustSelectExtension: boolean;
  mustSelectProjectFolderPath: boolean;
  nameError?: ProjectNameError;
  project?: RigProject;
  projectFilePath?: string;
  projectFolderPath: string;
  projectName: string;
  supportedViews: Set<ViewSupport>;
  versionError?: VersionError;
}

/*eslint dot-location: ["error", "object"]*/
export class CreateProjectWizard extends React.Component<PublicProps, State>{
  public state: State = {
    currentStep: 1,
    extensions: [],
    mustSelectCodeOption: false,
    mustSelectExtension: false,
    mustSelectProjectFolderPath: false,
    projectName: '',
    projectFolderPath: '',
    extensionVersion: initialExtensionVersion,
    supportedViews: new Set(),
  };

  public componentDidMount() {
    this.fetchExtensionClients();
  }

  public render() {
    const { closeHandler } = this.props;
    const {
      codeGeneration,
      currentStep,
      createProjectStatus,
      extensionGuide,
      extensionId,
      extensions,
      extensionVersion,
      isLoadingExtensions,
      isLoadingNextStep,
      mustSelectCodeOption,
      mustSelectExtension,
      mustSelectProjectFolderPath,
      nameError,
      projectName,
      projectFolderPath,
      supportedViews,
      versionError
    } = this.state;

    return (
      <Dialog canConfirm={!isLoadingNextStep}
        closeHandler={currentStep === finalStepNumber ? undefined : closeHandler}
        confirmText={currentStep === finalStepNumber ? createProjectStatus === CreateProjectStatus.Error ? 'OK' : 'Get Started' : 'Next'}
        confirmHandler={this.moveToNextStep}
        title={`Create New Project - Step ${currentStep} of ${finalStepNumber}`}
      >
        <Layout className="create-project-wizard__content">
          {currentStep === 1 ?
            <CreateProjectWizardStepOne
              extensions={extensions}
              isLoadingExtensions={isLoadingExtensions}
              mustSelectExtension={mustSelectExtension}
              nameError={nameError}
              onAddViewSupport={this.onAddExtensionViewSupport}
              onProjectNameChange={this.onProjectNameChange}
              onRefreshExtensionsClick={this.fetchExtensionClients}
              onRemoveViewSupport={this.onRemoveExtensionViewSupport}
              onSelectedExtensionChange={this.onSelectedExtensionChange}
              onSelectedVersionChange={this.onSelectedVersionChange}
              projectName={projectName}
              selectedExtensionId={extensionId}
              selectedExtensionVersion={extensionVersion}
              supportedViews={supportedViews}
              versionError={versionError}
            />
            : currentStep === 2 ?
              <CreateProjectWizardStepTwo
                mustSelectCodeOption={mustSelectCodeOption}
                mustSelectProjectFolderPath={mustSelectProjectFolderPath}
                onProjectFolderPathChange={this.onProjectFolderPathChange}
                onSelectedCodeGenerationChange={this.onSelectedCodeGenerationChange}
                onSelectedExtensionGuideChange={this.onSelectedExtensionGuideChange}
                projectFolderPath={projectFolderPath}
                selectedCodeGeneration={codeGeneration}
                selectedExtensionGuide={extensionGuide}
              />
              : currentStep === 3 &&
              <CreateProjectWizardStepThree
                createStatus={createProjectStatus}
                estimatedSaveDurationSeconds={extensionGuide ? extensionGuide.expectedDuration : 0}
              />}
        </Layout>
      </Dialog>
    );
  }

  private onProjectNameChange = (projectName: string) => {
    const nameError = validateProjectName(projectName);
    this.setState({
      nameError,
      projectName,
    });
  }

  private onProjectFolderPathChange = (projectFolderPath: string) => {
    this.setState({
      mustSelectProjectFolderPath: !projectFolderPath,
      projectFolderPath,
    });
  }

  private onSelectedExtensionChange = (extensionId: string) => {
    const extension = this.state.extensions.find((extension) => extension.id === extensionId);
    this.setState((prevState) => ({
      extensionId,
      extensionVersion: extensionId === NEW_EXTENSION_OPTION ? initialExtensionVersion : prevState.extensionVersion,
      mustSelectExtension: !extensionId,
      projectName: extension ? extension.name : '',
      nameError: extension ? undefined : InputError.EMPTY,
    }));
  }

  private onSelectedVersionChange = (extensionVersion: string) => {
    const versionError = validateVersion(extensionVersion);
    this.setState({
      extensionVersion,
      versionError,
    });
  }

  private onSelectedCodeGenerationChange = (codeGeneration: CodeGenerationOption) => {
    this.setState({
      codeGeneration,
      mustSelectCodeOption: false,
    });
  }

  private onSelectedExtensionGuideChange = (extensionGuide: Example) => {
    this.setState({
      extensionGuide,
    });
  }

  private onAddExtensionViewSupport = (type: ViewSupport) => {
    this.setState((prevState) => {
      const supportedViews = new Set(prevState.supportedViews);
      supportedViews.add(type);
      return {
        supportedViews,
      };
    });
  }

  private onRemoveExtensionViewSupport = (type: ViewSupport) => {
    this.setState((prevState) => {
      const supportedViews = new Set(prevState.supportedViews);
      supportedViews.delete(type);
      if (supportedViews.has(ViewSupport.Mobile) && supportedViews.size === 1) {
        supportedViews.delete(ViewSupport.Mobile);
      }
      return {
        supportedViews,
      };
    });
  }

  private moveToNextStep = async () => {
    const { currentStep } = this.state;
    this.setState({
      isLoadingNextStep: currentStep !== finalStepNumber,
    });
    let valid = true;
    if (currentStep === 1) {
      valid = await this.validateStepOne();
    } else if (currentStep === 2) {
      valid = this.validateStepTwo();
      if (valid) {
        this.createProject();
      }
    } else if (currentStep === finalStepNumber) {
      this.save();
      return;
    }

    this.setState((prevState) => {
      const current = prevState.currentStep;
      const nextStep = valid ? Math.min(current + 1, finalStepNumber) : current;
      return {
        currentStep: nextStep,
        isLoadingNextStep: false,
      };
    });
  }

  private async validateStepOne() {
    const { extensionVersion, nameError, projectName, supportedViews, versionError } = this.state;
    let valid = false;
    let extensionId = this.state.extensionId;

    if (!nameError && !projectName) {
      this.setState({ nameError: InputError.EMPTY });
    }
    if (!extensionId) {
      this.setState({ mustSelectExtension: true });
    }
    if (!extensionId || nameError || !projectName || versionError ||
      (extensionId === NEW_EXTENSION_OPTION && supportedViews.size < 1)) {
      return false;
    }

    try {
      if (extensionId === NEW_EXTENSION_OPTION) {
        try {
          extensionId = await this.createExtension();
        } catch (error) {
          const message: ProjectNameError = error.message;
          if (message === CreateExtensionClientError.INVALID_NAME || message === CreateExtensionClientError.NAME_IN_USE) {
            this.setState({
              nameError: error.message,
            });
            return false;
          } else {
            throw error;
          }
        }
      }

      // The extension secret may not be available immediately
      // after creating an extension so allow a retry
      const extensionSecret = await this.fetchExtensionSecret(extensionId, {
        maxAttempts: 3,
      });

      try {
        const extensionManifest = await fetchExtensionManifest(extensionId, extensionVersion);
        valid = true;
        this.setState({
          extensionId,
          extensionManifest,
          extensionSecret,
        });
      } catch (e) {
        const error: Error = e;
        if (error.message.includes('404')) {
          this.setState({
            versionError: VersionError.NotFound,
          });
        } else {
          throw e;
        }
      }
    } catch (error) {
      console.error(error);
      this.setState({ nameError: CreateExtensionClientError.NAME_IN_USE });
    }

    return valid;
  }

  private validateStepTwo() {
    const { projectFolderPath, codeGeneration, extensionGuide } = this.state;
    const isProjectFolderPathValid = projectFolderPath && electron.existsSync(projectFolderPath);
    if (!isProjectFolderPathValid) {
      this.setState({ mustSelectProjectFolderPath: true });
    }
    if (!codeGeneration) {
      this.setState({ mustSelectCodeOption: true });
    } else if (codeGeneration === CodeGenerationOption.Guide) {
      this.setState({ mustSelectCodeOption: !extensionGuide });
    }
    let valid = isProjectFolderPathValid && !!codeGeneration;
    if (valid && codeGeneration === CodeGenerationOption.Guide) {
      valid = !!extensionGuide;
    }
    return valid;
  }

  private fetchUserOrganizations = async (): Promise<string[]> => {
    let orgIDs: string[] = [];
    try {
      const resp = await graphqlClient.request<CurrentUserOrganizationsQueryResponse>(currentUserOrganizationsQuery)
      if (resp.currentUser && resp.currentUser.organizations) {
        orgIDs = resp.currentUser.organizations.map(orgInfo => orgInfo.id);
      }
    } finally {
      return orgIDs;
    }
  }

  private fetchExtensionClients = async () => {
    if (this.state.isLoadingExtensions) {
      return;
    }
    const orgIDs = await this.fetchUserOrganizations()
    this.setState({
      isLoadingExtensions: true,
    });
    const appliedOrgIds = [
      undefined, // this is for personal extensions
      ...orgIDs
    ];
    let extensionArray: Extension[] = [];
    for (const id of appliedOrgIds) {
      try {
        const { extensionClients } = await graphqlClient.request<ExtensionClientsQueryResponse>(extensionClientsQuery, {
          organizationID: id,
        });
        if (extensionClients) {
          const extensions = extensionClients.edges.
            sort((a, b) => Date.parse(a.node.createdAt) - Date.parse(b.node.createdAt)).
            map((e) => ({ id: e.node.id, name: this.parseExtensionName(e.node.name) }));
          extensionArray = [...extensionArray, ...extensions];
        }
      } catch (error) {
        // swallow error
      }
    };
    this.setState({
      extensions: extensionArray,
      isLoadingExtensions: false,
    });
  }

  private async createExtension() {
    const { id } = await this.createExtensionClient();
    await this.createExtensionManifest(id);
    return id;
  }

  private async createExtensionClient() {
    const { projectName } = this.state;
    if (!projectName) {
      throw new Error('Project name is required to create an Extension.');
    }

    const inputVariables: CreateExtensionClientMutationVariables = {
      extensionName: projectName,
    };
    const { createExtensionClient } = await graphqlClient.request<CreateExtensionClientMutationResponse>(createExtensionClientMutation, inputVariables);
    if (createExtensionClient) {
      const { client, error } = createExtensionClient;
      if (client) {
        return client;
      } else if (error) {
        throw new Error(error);
      } else {
        throw new Error('Extension client creation failed.');
      }
    } else {
      throw new Error('Extension client creation failed.');
    }
  }

  private async createExtensionManifest(extensionID: string) {
    const { extensionVersion, projectName, supportedViews } = this.state;
    const { rigLogin } = new LocalStorage();
    if (!rigLogin) {
      throw new Error('User session not available.');
    }

    const inputVariables = buildMutationInputForCreateExtensionManifest({
      authorName: rigLogin.displayName,
      authorEmail: rigLogin.email,
      extensionID,
      hasViewComponent: supportedViews.has(ViewSupport.VideoComponent),
      hasViewMobile: supportedViews.has(ViewSupport.Mobile),
      hasViewPanel: supportedViews.has(ViewSupport.Panel),
      hasViewVideoOverlay: supportedViews.has(ViewSupport.VideoOverlay),
      name: projectName,
      version: extensionVersion,
    });
    const { saveExtensionManifest } = await graphqlClient.request<SaveExtensionManifestMutationResponse>(createExtensionManifestMutation, inputVariables);
    if (saveExtensionManifest) {
      const { error, manifest } = saveExtensionManifest;
      if (manifest) {
        return manifest;
      } else if (error) {
        throw error;
      }
    }
    throw new Error('Extension manifest creation failed.');
  }

  private async fetchExtensionSecret(extensionId: string, options: { maxAttempts: number }): Promise<string> {
    const { maxAttempts } = options;
    if (maxAttempts <= 0) {
      throw new Error(`Could not retrieve extension secret for extension ${extensionId}.`);
    }

    try {
      return await fetchExtensionSecret(extensionId);
    } catch (e) {
      const error: Error = e;
      if (error.message.includes('404')) {
        await timeout(550);
        return await this.fetchExtensionSecret(extensionId, { maxAttempts: maxAttempts - 1 });
      } else {
        throw e;
      }
    }
  }

  private parseExtensionName(rawName: string) {
    const name = extensionNameRegex.exec(rawName);
    if (name && name[1]) {
      return name[1];
    }
    return rawName;
  }

  private async createProject() {
    const {
      codeGeneration,
      extensionGuide,
      extensionManifest,
      extensionSecret,
      projectFolderPath,
      projectName
    } = this.state;
    let extensionGuideId: number | undefined;

    if (!extensionManifest || !codeGeneration) {
      return;
    }

    const project: RigProject = {
      allowHttpBackend: false,
      backendCommand: '',
      backendFolderName: '',
      certificateExceptions: [],
      extensionViews: [],
      manifest: extensionManifest,
      frontendCommand: '',
      frontendFolderName: '',
      name: projectName,
      projectFolderPath: projectFolderPath || '',
      secret: extensionSecret || '',
      usingRandomFrontendHostingPort: false,
      version: 3,
    };

    if (codeGeneration !== CodeGenerationOption.None && extensionGuide) {
      extensionGuideId = extensionGuide.id;
      project.backendCommand = this.constructBackendCommand(extensionGuide);
      project.backendFolderName = extensionGuide.backendFolderName || '';
      project.frontendCommand = extensionGuide.frontendCommand;
      project.frontendFolderName = extensionGuide.frontendFolderName;
    }

    const codeType = codeGeneration === CodeGenerationOption.None ? 'none' : 'example';
    if (codeGeneration === CodeGenerationOption.Boilerplate) {
      const examples = await fetchExamples();
      extensionGuideId = examples.find((example) => !example.isGuide)!.id;
    }
    this.setState({
      createProjectStatus: CreateProjectStatus.Loading,
    });

    let projectFilePath: string | undefined;
    try {
      projectFilePath = await createProject(codeType, project, extensionGuideId);
    } catch (e) {
      console.error(e);
      this.setState({
        createProjectStatus: CreateProjectStatus.Error,
      });
    }

    if (projectFilePath) {
      this.setState({
        createProjectStatus: CreateProjectStatus.Success,
        project,
        projectFilePath,
      });
    }
  }

  private constructBackendCommand(example: Example) {
    const { extensionSecret, codeGeneration: selectedCodeGeneration, extensionId } = this.state;
    if (extensionId &&
      extensionSecret &&
      selectedCodeGeneration === CodeGenerationOption.Guide && example.backendCommand) {
      const backendCommand = example.backendCommand.
        replace('{clientId}', extensionId).
        replace('{secret}', extensionSecret).
        replace('{ownerId}', this.props.userId);
      return backendCommand;
    }
    return '';
  }

  private save() {
    const { createProjectStatus, project, projectFilePath } = this.state;
    if (project && projectFilePath) {
      this.props.saveHandler(project, projectFilePath, project.secret);
    } else if (createProjectStatus === CreateProjectStatus.Error) {
      this.props.closeHandler();
    }
  }
}
