import * as React from 'react';
import { Select, FormGroup, Input, InputType, Display, SVGAsset, Button, ButtonType, ButtonState, ImageSelector, BorderRadius, ImageSelectorType, Layout, CoreText, TextAlign, Color } from 'twitch-core-ui';
import { VersionError, messageForVersionError } from '../../utils/validate-version';
import { ProjectNameError, messageForProjectNameError } from '../../utils/validate-project-name';
import iconPanelExtension from '../../../img/icon-panel-extension.svg';
import iconOverlayExtension from '../../../img/icon-overlay-extension.svg';
import iconComponentExtension from '../../../img/icon-component-extension.svg';
import iconMobileExtension from '../../../img/icon-mobile-extension.svg';
import { sendEvent } from '../../../util/api';

export const NEW_EXTENSION_OPTION = 'new';

export enum ViewSupport {
  Panel = 'view-support-panel',
  VideoOverlay = 'view-support-overlay',
  VideoComponent = 'view-support-component',
  Mobile = 'view-support-mobile',
}

interface PublicProps {
  extensions: {
    id: string;
    name: string;
  }[];
  isLoadingExtensions?: boolean;
  mustSelectExtension: boolean;
  nameError?: ProjectNameError;
  onAddViewSupport: (type: ViewSupport) => void;
  onProjectNameChange: (name: string) => void;
  onRefreshExtensionsClick: () => void;
  onRemoveViewSupport: (type: ViewSupport) => void;
  onSelectedExtensionChange: (selectedExtensionID: string) => void;
  onSelectedVersionChange: (selectedVersion: string) => void;
  projectName: string;
  selectedExtensionId?: string;
  selectedExtensionVersion: string;
  supportedViews: Set<ViewSupport>;
  versionError?: VersionError;
}

export class CreateProjectWizardStepOne extends React.Component<PublicProps>{
  componentDidMount() {
    sendEvent('dx_rig_create_funnel_step', { step_label: 'pick-extension' });
  }

  public render() {
    const {
      extensions,
      isLoadingExtensions,
      mustSelectExtension,
      nameError,
      onRefreshExtensionsClick,
      projectName,
      selectedExtensionId,
      selectedExtensionVersion,
      supportedViews,
      versionError
    } = this.props;
    let projectNamePlaceholder = 'Enter a name for your project.';
    if (selectedExtensionId === NEW_EXTENSION_OPTION) {
      projectNamePlaceholder += ' This will also be the name of your extension.';
    }
    const mobileSupportDisabled = supportedViews.size < 1;

    return (
      <>
        <FormGroup error={mustSelectExtension}
          label="Select Existing or Create New Extension"
          required errorMessage="Please select an extension or create a new one."
        >
          <Layout display={Display.Flex}>
            <Layout flexGrow={1}>
              <Select onChange={this.onSelectedExtensionChange} required value={selectedExtensionId}>
                <option value="">Select</option>
                <option value={NEW_EXTENSION_OPTION}>New Extension</option>
                {extensions.map((e) => <option key={e.id} value={e.id}>{e.name}</option>)}
              </Select>
            </Layout>
            <Layout padding={{ left: .5 }}>
              <Button
                ariaLabel="CreateProjectWizardStepOne:Refresh"
                icon={SVGAsset.Refresh}
                onClick={onRefreshExtensionsClick}
                state={isLoadingExtensions ? ButtonState.Loading : ButtonState.Default}
                type={ButtonType.Hollow}
              />
            </Layout>
          </Layout>
        </FormGroup>
        <Layout padding={{ top: 2 }}>
          <FormGroup
            error={!!nameError}
            errorMessage={messageForProjectNameError(nameError)}
            label="Project Name"
            required
          >
            <Input
              onChange={this.onProjectNameChange}
              placeholder={projectNamePlaceholder}
              required
              type={InputType.Text}
              value={projectName}
            />
          </FormGroup>
        </Layout>
        <Layout padding={{ top: 2 }}>
          <FormGroup
            error={!!versionError}
            errorMessage={messageForVersionError(versionError)}
            hint="Extensions can have multiple independent versions for building and release. You can add versions or change to new ones as you continue development."
            label="Extension Version"
            required
          >
            <Input
              error={!!versionError}
              onChange={this.onSelectedVersionChange}
              placeholder="Enter your extension's version"
              required
              type={InputType.Text}
              value={selectedExtensionVersion}
            />
          </FormGroup>
        </Layout>
        {selectedExtensionId === NEW_EXTENSION_OPTION && (
          <Layout padding={{ top: 2 }}>
            <FormGroup
              label="Select Extension Types"
              required
            >
              <Layout
                display={Display.Flex}
              >
                <Layout textAlign={TextAlign.Center}>
                  <ImageSelector
                    alt="Panel"
                    borderRadius={BorderRadius.Medium}
                    checked={supportedViews.has(ViewSupport.Panel)}
                    label="Panel"
                    name="extension-type-selector"
                    onChange={this.onSelectedExtensionTypesChange}
                    selectedIcon
                    src={iconPanelExtension}
                    type={ImageSelectorType.Checkbox}
                    value={ViewSupport.Panel}
                  />
                  <CoreText>Panel</CoreText>
                </Layout>
                <Layout padding={{ left: 1 }} textAlign={TextAlign.Center}>
                  <ImageSelector
                    alt="Video Fullscreen"
                    borderRadius={BorderRadius.Medium}
                    checked={supportedViews.has(ViewSupport.VideoOverlay)}
                    name="extension-type-selector"
                    label="Video Fullscreen"
                    onChange={this.onSelectedExtensionTypesChange}
                    selectedIcon
                    src={iconOverlayExtension}
                    type={ImageSelectorType.Checkbox}
                    value={ViewSupport.VideoOverlay}
                  />
                  <CoreText>Video Fullscreen</CoreText>
                </Layout>
                <Layout padding={{ left: 1 }} textAlign={TextAlign.Center}>
                  <ImageSelector
                    alt="Video Component"
                    borderRadius={BorderRadius.Medium}
                    checked={supportedViews.has(ViewSupport.VideoComponent)}
                    label="Video Component"
                    name="extension-type-selector"
                    onChange={this.onSelectedExtensionTypesChange}
                    src={iconComponentExtension}
                    selectedIcon
                    type={ImageSelectorType.Checkbox}
                    value={ViewSupport.VideoComponent}
                  />
                  <CoreText>Video Component</CoreText>
                </Layout>
                <Layout padding={{ left: 1 }} textAlign={TextAlign.Center}>
                  <ImageSelector
                    alt="Mobile"
                    borderRadius={BorderRadius.Medium}
                    checked={supportedViews.has(ViewSupport.Mobile)}
                    disabled={mobileSupportDisabled}
                    label="Mobile"
                    name="extension-type-selector"
                    onChange={this.onSelectedExtensionTypesChange}
                    src={iconMobileExtension}
                    selectedIcon
                    type={ImageSelectorType.Checkbox}
                    value={ViewSupport.Mobile}
                  />
                  <CoreText
                    color={mobileSupportDisabled ? Color.OverlayAlt : undefined}
                  >
                    Mobile
                  </CoreText>
                </Layout>
              </Layout>
            </FormGroup>
          </Layout>
        )}
      </>
    );
  }

  private onProjectNameChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const name = event.target.value;
    this.props.onProjectNameChange(name);
  }

  private onSelectedExtensionChange = (event: React.ChangeEvent<HTMLSelectElement>) => {
    const selectedExtensionId = event.target.value;
    this.props.onSelectedExtensionChange(selectedExtensionId);
  }

  private onSelectedVersionChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const selectedExtensionVersion = event.target.value;
    this.props.onSelectedVersionChange(selectedExtensionVersion);
  }

  private onSelectedExtensionTypesChange = (event: React.ChangeEvent<HTMLInputElement>) => {
    const { checked, value } = event.target;
    const { onAddViewSupport: onAddExtensionViewSupport, onRemoveViewSupport: onRemoveExtensionViewSupport } = this.props;
    const type = value as ViewSupport;
    checked ? onAddExtensionViewSupport(type) : onRemoveExtensionViewSupport(type);
  }
}
