import * as React from 'react';
import { FormGroup, Layout, Input, InputType, Button, ButtonType, Display, DropDownMenu, BalloonSize, DropDownMenuItem, CoreText } from 'twitch-core-ui';
import { fetchExamples, Example, showOpenDialog, sendEvent, hasNode } from '../../../util/api';
import { makeCancelablePromise, CancelablePromise } from '../../../util/make-cancelable-promise';
import { ExtensionGuideCard } from './components/extension-guide-card';
import { ToggleBalloonWrapper } from '../../../core/components';
import { Dialog } from '../../../dialog';

export enum CodeGenerationOption {
  None = 'none',
  Boilerplate = 'boilerplate',
  Guide = 'guide',
}

interface PublicProps {
  mustSelectCodeOption: boolean;
  mustSelectProjectFolderPath: boolean;
  onProjectFolderPathChange: (path: string) => void;
  onSelectedCodeGenerationChange: (option: CodeGenerationOption) => void;
  onSelectedExtensionGuideChange: (guide: Example) => void;
  projectFolderPath: string;
  selectedCodeGeneration?: CodeGenerationOption;
  selectedExtensionGuide?: Example;
}

interface State {
  extensionGuides: Example[];
  hasNode: boolean;
  showingNodeRequirement: boolean;
}

export class CreateProjectWizardStepTwo extends React.Component<PublicProps, State>{
  public state: State = {
    extensionGuides: [],
    hasNode: false,
    showingNodeRequirement: false,
  };
  private fetchExamplesCancelable?: CancelablePromise<Example[]>;
  private dropdownToggle: ToggleBalloonWrapper | null = null;

  public async componentDidMount() {
    await this.fetchExamples();
    sendEvent('dx_rig_create_funnel_step', { step_label: 'choose-folder' });
    this.setState({ hasNode: await hasNode() });
  }

  public componentWillUnmount() {
    if (this.fetchExamplesCancelable) {
      this.fetchExamplesCancelable.cancel();
    }
  }

  private onProjectFolderPathChange = (event: React.FormEvent<HTMLInputElement>) => {
    const { value } = event.currentTarget;
    this.props.onProjectFolderPathChange(value);
  }

  public render() {
    const { mustSelectCodeOption, mustSelectProjectFolderPath, projectFolderPath, selectedCodeGeneration, selectedExtensionGuide } = this.props;
    const extensionGuideOptionLabel = 'Use Extension tutorial example (Recommended for new developers)';
    const boilerplateOptionLabel = 'Use boilerplate code (Preferred for extensions using React)';
    const noneOptionLabel = "None - I'll use my own code";
    const nodeRemediationText = window.clientInformation.platform === 'Win32' ?
      'make sure it is in your system PATH environment variable' :
      'create a symbolic link to it in /usr/local/bin';
    return (
      <>
        <FormGroup error={mustSelectProjectFolderPath}
          errorMessage="You must select a local folder."
          label="Select the local folder where your extensions files will be located during development."
        >
          <Layout display={Display.Flex}>
            <Layout flexGrow={1} padding={{ right: 1 }}>
              <Input error={mustSelectProjectFolderPath}
                placeholder="Select folder"
                type={InputType.Text}
                value={projectFolderPath}
                onChange={this.onProjectFolderPathChange}
              />
            </Layout>
            <Button ariaLabel="CreateProjectWizardStepTwo:Select" type={ButtonType.Hollow} onClick={this.selectProjectFolder}>Select</Button>
          </Layout>
        </FormGroup>
        <Layout padding={{ top: 2 }}>
          <FormGroup error={mustSelectCodeOption} errorMessage="Please select the code you'll use for your extension." label="Add Code to your Project">
            <ToggleBalloonWrapper ref={(element) => this.dropdownToggle = element}>
              <Button ariaLabel="CreateProjectWizardStepTwo:Select template" dropdown type={mustSelectCodeOption && !selectedCodeGeneration ? ButtonType.Alert : ButtonType.Hollow}>
                {!selectedCodeGeneration && 'Select template'}
                {selectedCodeGeneration === CodeGenerationOption.Guide && extensionGuideOptionLabel}
                {selectedCodeGeneration === CodeGenerationOption.Boilerplate && boilerplateOptionLabel}
                {selectedCodeGeneration === CodeGenerationOption.None && noneOptionLabel}
              </Button>
              <DropDownMenu
                noTail
                offsetY="-1px"
                size={BalloonSize.ExtraLarge}
              >
                <DropDownMenuItem title={extensionGuideOptionLabel} onClick={this.onExtensionGuideOptionClick} />
                <DropDownMenuItem title={boilerplateOptionLabel} onClick={this.onBoilerplateOptionClick} />
                <DropDownMenuItem title={noneOptionLabel} onClick={this.onNoneOptionClick} />
              </DropDownMenu>
            </ToggleBalloonWrapper>
          </FormGroup>
          {selectedCodeGeneration === CodeGenerationOption.Guide &&
            <Layout padding={{ top: 1 }}>
              {this.state.extensionGuides.filter((example) => example.isGuide).map((example) => (
                <Layout key={example.id} padding={{ top: 1 }}>
                  <ExtensionGuideCard
                    description={example.description}
                    linkUrl={`https://github.com/${example.repository}`}
                    selected={!!selectedExtensionGuide && selectedExtensionGuide.title === example.title}
                    title={example.title}
                    onClick={this.onExtensionGuideExampleClick}
                  />
                </Layout>
              ), [])}
            </Layout>
          }
          {this.state.showingNodeRequirement && <Dialog canConfirm={true} confirmHandler={() => this.setState({ showingNodeRequirement: false })} confirmText="OK" title="Node Not Available">
            <div style={{ display: 'grid', gridGap: '8px' }}>
              <CoreText>The Developer Rig cannot find an installation of Node to support this selection.</CoreText>
              <CoreText>If you already have it installed, {nodeRemediationText} and restart the Developer Rig.</CoreText>
              <CoreText>Otherwise, you may download it from <a aria-label="CreateProjectWizardStepTwo:download Node" href="https://nodejs.org" rel="noopener noreferrer" target="_blank">here</a>.</CoreText>
            </div>
          </Dialog>}
        </Layout>
      </>
    );
  }

  private selectProjectFolder = async () => {
    const projectFolderPath = await showOpenDialog({ properties: ['openDirectory', 'createDirectory'] });
    if (projectFolderPath) {
      this.props.onProjectFolderPathChange(projectFolderPath);
      sendEvent('dx_rig_create_funnel_step', { step_label: 'add-code' });
    }
  }

  private async fetchExamples() {
    this.fetchExamplesCancelable = makeCancelablePromise(fetchExamples());
    const extensionGuides = await this.fetchExamplesCancelable.promise;
    this.setState({
      extensionGuides,
    });
  }

  private onExtensionGuideOptionClick = () => {
    this.setSelectedCodeGeneration(CodeGenerationOption.Guide);
  }

  private onBoilerplateOptionClick = () => {
    this.setSelectedCodeGeneration(CodeGenerationOption.Boilerplate);
  }

  private onNoneOptionClick = () => {
    this.setSelectedCodeGeneration(CodeGenerationOption.None);
  }

  private setSelectedCodeGeneration(selectedCodeGeneration: CodeGenerationOption) {
    if (this.dropdownToggle) {
      this.dropdownToggle.toggleBalloon(false);
    }
    if (selectedCodeGeneration === CodeGenerationOption.None || this.state.hasNode) {
      this.props.onSelectedCodeGenerationChange(selectedCodeGeneration);
    } else {
      this.setState({ showingNodeRequirement: true });
    }
  }

  private onExtensionGuideExampleClick = (guideTitle: string) => {
    const guide = this.state.extensionGuides.find((g) => g.title === guideTitle);
    if (guide) {
      this.props.onSelectedExtensionGuideChange(guide);
    }
  };
}
