import { CreateExtensionClientError } from "../queries/models";

const maximumLength = 40;
const validNameRegEx = new RegExp(/(twitch|extension|glitch)/mig);

type ServerError = CreateExtensionClientError.NAME_IN_USE | CreateExtensionClientError.INVALID_NAME;
export enum InputError {
  EMPTY = 'input-empty-error',
}
export type ProjectNameError = InputError | ServerError;

export function validateProjectName(name: string) {
  if (name.length === 0) {
    return InputError.EMPTY;
  }
  if (name.length > maximumLength) {
    return CreateExtensionClientError.INVALID_NAME;
  }
  if (validNameRegEx.test(name)) {
    return CreateExtensionClientError.INVALID_NAME;
  }
}

export function messageForProjectNameError(error?: ProjectNameError) {
  let errorMessage: string | undefined;
  switch (error) {
    case CreateExtensionClientError.NAME_IN_USE:
      errorMessage = 'Sorry, this name is taken. Please try another name. Note: Twitch broadcasters installing your Extension see the name specified in the released version. This is only for API authorization use.';
      break;
    case CreateExtensionClientError.INVALID_NAME:
      errorMessage = `Please confirm that the name does not include the words Twitch, Glitch, or Extension, and that it is no more than ${maximumLength} characters.`;
      break;
    case InputError.EMPTY:
      errorMessage = 'Please name your project.';
      break;
  }
  return errorMessage;
}
