const maximumLength = 40;
const versionFormatRegex = /^\d+(\.\d+)?(\.\d+)?$/;

export enum VersionError {
  Duplicate = 'duplicate',
  Empty = 'empty',
  Format = 'format',
  Length = 'length',
  NotFound = 'not-found',
}

export function validateVersion(version: string) {
  if (version.length === 0) {
    return VersionError.Empty;
  }
  if (version && !versionFormatRegex.test(version)) {
    return VersionError.Format;
  }
  if (version && version.length > maximumLength) {
    return VersionError.Length;
  }
}

export function messageForVersionError(error?: VersionError) {
  let errorMessage: string | undefined;
  switch (error) {
    case VersionError.Empty:
      errorMessage = 'Please choose a correct version you have created. Ex. 0.0.1';
      break;
    case VersionError.Format:
      errorMessage = 'Version numbers must be in the format x.x.x and contain only numbers.';
      break;
    case VersionError.Length:
      errorMessage = `Version numbers have a ${maximumLength} character limit.`;
      break;
    case VersionError.Duplicate:
      errorMessage = 'This version number has already been used.';
      break;
    case VersionError.NotFound:
      errorMessage = "There is no such version for your extension.";
  }
  return errorMessage;
}
