import * as React from 'react';
import './component.sass';
import classNames from 'classnames';
import { Layout, StyledLayout, Background, Position, ZIndex, Display, AlignItems, JustifyContent, Modal, ModalSize, Color } from 'twitch-core-ui';

interface Props extends React.Props<{}> {
  className?: string;
  canConfirm: boolean;
  confirmText: string;
  title: string;
  closeHandler?: () => void;
  confirmHandler: () => void;
}

export class Dialog extends React.Component<Props> {
  componentDidMount() {
    document.addEventListener('keydown', this.handleKeyDown);
  }

  componentWillUnmount() {
    document.removeEventListener('keydown', this.handleKeyDown);
  }

  private handleKeyDown = (event: KeyboardEvent) => {
    if (event.key === 'Escape' && this.props.closeHandler) {
      this.props.closeHandler();
    } else if (event.key === 'Enter' && this.props.canConfirm && this.props.confirmHandler) {
      this.props.confirmHandler();
    }
  }

  public render() {
    const { className, canConfirm, children, confirmText, title, closeHandler, confirmHandler } = this.props;
    const bodyClassNames = classNames('dialog__body', className);
    return <>
      <Layout alignItems={AlignItems.Center}
        attachLeft attachTop
        className="dialog" display={Display.Flex}
        fullHeight fullWidth
        justifyContent={JustifyContent.Center} position={Position.Fixed} zIndex={ZIndex.Above}>
        <StyledLayout
          attachBottom attachLeft attachRight attachTop
          background={Background.Overlay}
          fullHeight fullWidth
          position={Position.Fixed} zIndex={ZIndex.Above} />
        <StyledLayout color={Color.Base} zIndex={ZIndex.Above}>
          <Modal
            headerProps={{
              closeButton: closeHandler ? {
                ariaLabel: `${title}:Close`,
                onClick: closeHandler,
              } : undefined,
              title,
            }}
            footerProps={{
              primaryButtonProps: {
                ariaLabel: `${title}:${confirmText}`,
                children: confirmText,
                disabled: !canConfirm,
                onClick: confirmHandler,
              },
              secondaryButtonProps: closeHandler ? {
                ariaLabel: `${title}:Cancel`,
                children: 'Cancel',
                onClick: closeHandler,
              } : undefined,
            }}
            size={ModalSize.Large}
          >
            <div className={bodyClassNames}>
              {children}
            </div>
          </Modal>
        </StyledLayout>
      </Layout>
    </>;
  }
}
