import { setupShallowTest } from '../tests/enzyme-util/shallow';
import { createViewsForTest, createExtensionManifestForTest } from '../tests/constants/extension';
import { ExtensionViewContainer } from './component';
import { ViewerType } from '../core/models/rig';
import { ExtensionAnchor } from 'extension-coordinator';
import { timeout } from '../tests/utils';
import { MobileOrientation } from '../constants/mobile';

function mockApiFunctions() {
  return {
    ...require.requireActual('../util/api'),
    fetchUser: jest.fn().mockImplementation(() => Promise.resolve({ id: 'id' })),
    fetchHostingStatus: jest.fn().mockImplementation(() => Promise.resolve({ isBackendRunning: true })),
  };
}
jest.mock('../util/api', () => mockApiFunctions());
const api = require.requireMock('../util/api');

function mockRigFunctions() {
  return {
    ...require.requireActual('../core/models/rig'),
    toggleBackend: jest.fn(() => Promise.resolve('test')),
    toggleFrontend: jest.fn(() => Promise.resolve('test')),
  };
}
jest.mock('../core/models/rig', () => mockRigFunctions());
const rig = require.requireMock('../core/models/rig');

const rigProject = {
  allowHttpBackend: false,
  backendCommand: 'backendCommand',
  backendFolderName: 'backendFolderName',
  certificateExceptions: [],
  extensionViews: createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.Broadcaster),
  frontendCommand: 'frontendCommand',
  frontendFolderName: 'frontendFolderName',
  manifest: createExtensionManifestForTest(),
  name: 'name',
  projectFolderPath: 'projectFolderPath',
  secret: 'secret',
  version: 1,
};
const setupShallow = setupShallowTest(ExtensionViewContainer, () => ({
  authToken: 'authToken',
  isDisplayed: true,
  isConsoleWide: true,
  rigProject,
  onCreateExtensionView: jest.fn(() => Promise.resolve()),
  onDeleteExtensionView: jest.fn(),
  onUpdateExtensionViews: jest.fn(),
  onViewInBrowser: jest.fn(),
}));

describe('<ExtensionViewContainer />', () => {
  it('opens, creates, and closes ExtensionViewDialog', async () => {
    api.fetchUser.mockClear();
    const { wrapper, props } = setupShallow();
    wrapper.find('Button').first().simulate('click');
    expect(wrapper.state('showingExtensionsViewDialog')).toBe(true);
    expect(wrapper.find('ExtensionViewDialog').length).toBe(1);
    await wrapper.find('ExtensionViewDialog').get(0).props.saveHandler({});
    await timeout();
    expect(wrapper.state('showingExtensionsViewDialog')).toBe(false);
    expect(props.onCreateExtensionView).toHaveBeenCalled();
    expect(api.fetchUser).toHaveBeenCalled();
  });

  it('toggles back end', async () => {
    const { wrapper } = setupShallow();
    wrapper.find('Button').at(2).simulate('click');
    expect(rig.toggleBackend).toHaveBeenCalled();
  });

  it('toggles front end', async () => {
    const { wrapper } = setupShallow();
    wrapper.find('Button').at(1).simulate('click');
    expect(rig.toggleFrontend).toHaveBeenCalled();
  });

  it('unmounts', async () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    wrapper.unmount();
    expect(instance.isActive).toBeFalsy();
  });

  it('creates custom dialog size', async () => {
    api.fetchUser.mockClear();
    const { wrapper, props } = setupShallow();
    wrapper.find('Button').first().simulate('click');
    expect(wrapper.state('showingExtensionsViewDialog')).toBe(true);
    expect(wrapper.find('ExtensionViewDialog').length).toBe(1);
    await wrapper.find('ExtensionViewDialog').get(0).props.saveHandler({ frameSize: 'Custom' });
    await timeout();
    expect(wrapper.state('showingExtensionsViewDialog')).toBe(false);
    expect(props.onCreateExtensionView).toHaveBeenCalled();
    expect(api.fetchUser).toHaveBeenCalled();
  });

  it('does not show extension view dialog', async () => {
    const { wrapper } = setupShallow({ isDisplayed: false });
    expect(wrapper.state('showingExtensionsViewDialog')).toBe(false);
  });

  it('opens, saves, and closes EditViewDialog', () => {
    const extensionViews = createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.LoggedOut);
    const { wrapper } = setupShallow({ rigProject: Object.assign({}, rigProject, { extensionViews }) });
    wrapper.find('ExtensionView').get(0).props.onOpenEditView('1');
    expect(wrapper.state('viewForEdit')).toBe(extensionViews[0]);
    expect(wrapper.find('EditViewDialog').length).toBe(1);
    wrapper.find('EditViewDialog').get(0).props.updateViewHandler(MobileOrientation.Portrait);
    expect(wrapper.state('viewForEdit')).not.toBeDefined();
  });

  it('handles console minimization', () => {
    const { wrapper } = setupShallow();
    wrapper.find('Console').simulate('minimize', true);
    expect(wrapper.state('isConsoleMinimized')).toBe(true);
  });
});
