import * as React from 'react';
import './component.sass';
import classNames from 'classnames';
import { RigProject, HostingResult, toggleBackend, toggleFrontend } from '../core/models/rig';
import { Console } from './console';
import { ExtensionMode } from 'extension-coordinator';
import { ExtensionView } from './extension-view';
import { RigExtensionView } from '../core/models/rig';
import { createExtensionObject } from '../util/extension';
import { ExtensionViewDefinition, ExtensionViewDialog } from './extension-view-dialog';
import { EditViewDialog } from './edit-view-dialog';
import { fetchIdForUser } from '../util/id';
import { IdentityOption } from '../core/models/identity-option';
import { MobileOrientation } from '../constants/mobile';
import { sendEvent, fetchHostingStatus } from '../util/api';
import { Button, CoreText, TextType, SVG, SVGAsset, SVGType } from 'twitch-core-ui';

interface Props {
  authToken: string;
  isDisplayed: boolean;
  isConsoleWide: boolean;
  rigProject: RigProject;
  onCreateExtensionView: (extensionView: RigExtensionView) => Promise<void>;
  onDeleteExtensionView: (viewId: string) => void;
  onUpdateExtensionViews: (extensionViews: RigExtensionView[]) => void;
  onViewInBrowser: (viewId: string) => void;
}

interface State {
  backendResult: string;
  onDialogClosed?: () => void;
  frontendResult: string;
  isConsoleMinimized: boolean;
  refreshCount: number;
  showingExtensionsViewDialog: boolean;
  viewForEdit?: RigExtensionView;
}

export class ExtensionViewContainer extends React.Component<Props, State> {
  state: State = {
    backendResult: HostingResult.None,
    frontendResult: HostingResult.None,
    isConsoleMinimized: false,
    refreshCount: 0,
    showingExtensionsViewDialog: false,
  };
  private isActive: boolean = false;

  static getDerivedStateFromProps(props: Props, _state: State): Partial<State> | null {
    if (!props.isDisplayed) {
      return { showingExtensionsViewDialog: false, viewForEdit: undefined };
    }
    return null;
  }

  async componentDidMount() {
    this.isActive = true;
    const status = await fetchHostingStatus();
    if (this.isActive) {
      const { rigProject } = this.props;
      const getStatus = (value: string, isRunning: boolean): HostingResult => {
        return value ? isRunning ? HostingResult.Running : HostingResult.NotRunning : HostingResult.None;
      };
      this.setState({
        backendResult: getStatus(rigProject.backendCommand, status.isBackendRunning),
        frontendResult: status.isFrontendRunning ? HostingResult.Running : HostingResult.NotRunning,
      });
    }
  }

  componentWillUnmount() {
    this.isActive = false;
  }

  private openEditViewDialog = (viewId: string): Promise<void> => {
    return new Promise<void>((resolve, _reject) => {
      this.setState({
        onDialogClosed: resolve,
        viewForEdit: this.props.rigProject.extensionViews.filter((extensionView) => extensionView.id === viewId)[0],
      });
    });
  }

  private updateView = (orientation?: MobileOrientation) => {
    const viewForEdit = this.state.viewForEdit!;
    viewForEdit.orientation = orientation;
    this.props.onUpdateExtensionViews(this.props.rigProject.extensionViews);
    this.closeEditViewDialog();
  }

  private closeEditViewDialog = () => {
    this.state.onDialogClosed!();
    this.setState({ onDialogClosed: undefined, viewForEdit: undefined });
  }

  private openExtensionViewDialog = () => {
    this.setState({ showingExtensionsViewDialog: true });
  }

  private createExtensionView = async (definition: ExtensionViewDefinition) => {
    const { authToken, rigProject: { manifest }, onCreateExtensionView } = this.props;
    let { channelId, configurationAnchor, displayChannelName, extensionType } = definition;
    channelId = await fetchIdForUser(authToken, channelId);
    const linked = definition.identityOption === IdentityOption.Linked ||
      extensionType === ExtensionMode.Config ||
      extensionType === ExtensionMode.Dashboard;
    const linkedUserId = linked && definition.linkedUserId ?
      await fetchIdForUser(authToken, definition.linkedUserId) : '';
    const extensionView: RigExtensionView = {
      channelId,
      configurationAnchor,
      displayChannelName,
      features: { ...definition.features },
      frameSize: definition.frameSize,
      id: '',
      isPopout: definition.isPopout,
      label: definition.label,
      linked,
      linkedUserId,
      opaqueId: definition.opaqueId,
      orientation: definition.orientation,
      type: extensionType,
      viewerType: definition.viewerType,
    };
    await onCreateExtensionView(extensionView);
    this.closeExtensionViewDialog();
    const eventData = {
      channel_id: channelId,
      extension_id: manifest.id,
      extension_version: manifest.version,
      view_medadata: JSON.stringify(definition),
      view_type: extensionType,
      viewer_type: definition.viewerType,
    };
    sendEvent('dx_rig_create_view', eventData);
  }

  private closeExtensionViewDialog = () => {
    this.setState({ showingExtensionsViewDialog: false });
  }

  private toggleBackend = async () => {
    const backendState = this.state.backendResult;
    this.setState({ backendResult: HostingResult.None });
    const backendResult = await toggleBackend(this.props.rigProject, backendState);
    if (backendResult) {
      this.setState({ backendResult });
      this.refreshViews();
    }
  }

  private toggleFrontend = async () => {
    const frontendState = this.state.frontendResult;
    this.setState({ frontendResult: HostingResult.None });
    const frontendResult = await toggleFrontend(this.props.rigProject, frontendState);
    if (frontendResult) {
      this.setState({ frontendResult });
      this.refreshViews();
    }
  }

  private getIsRunning(result: string) {
    return result === HostingResult.Started || result === HostingResult.Running;
  }

  private refreshViews = () => {
    this.setState((previousState) => ({ refreshCount: previousState.refreshCount + 1 }));
  }

  public render() {
    const { isConsoleWide, isDisplayed, rigProject, onDeleteExtensionView, onViewInBrowser } = this.props;
    const { backendResult, frontendResult, refreshCount } = this.state;
    const className = classNames('extension-view-container', {
      'extension-view-container--console-minimized': this.state.isConsoleMinimized,
      'extension-view-container--hidden': !isDisplayed,
    });
    const isRemoteFrontEnd = !rigProject.frontendCommand && !rigProject.frontendFolderName;
    const isFrontendRunning = this.getIsRunning(frontendResult);
    const frontendCommandText = isFrontendRunning ? 'Stop Front End' : 'Run Front End';
    const isBackendRunning = this.getIsRunning(backendResult);
    const backendCommandText = isBackendRunning ? 'Stop Back End' : 'Run Back End';
    return <>
      <div className={className}>
        <div className="extension-view-container__button-bar">
          <CoreText type={TextType.H3}>Extension Views</CoreText>
          <Button ariaLabel="ExtensionViewContainer:Create New View" onClick={this.openExtensionViewDialog}>
            <div className="extension-view-container__button">
              <SVG asset={SVGAsset.AddReaction} type={SVGType.Inherit} />
              <CoreText type={TextType.Span}>Create New View</CoreText>
            </div>
          </Button>
          {rigProject.manifest.state === 'Testing' && <Button ariaLabel={`ExtensionViewContainer:${frontendCommandText}`} disabled={isRemoteFrontEnd} onClick={this.toggleFrontend}>{frontendCommandText}</Button>}
          {rigProject.backendCommand && <Button ariaLabel={`ExtensionViewContainer:${backendCommandText}`} onClick={this.toggleBackend}>{backendCommandText}</Button>}
          <Button ariaLabel="ExtensionViewContainer:Refresh All Views" onClick={this.refreshViews}>Refresh All Views</Button>
        </div>
        <div className="extension-view-container__views">
          {rigProject.extensionViews.map((view, index) => {
            const linkedUserId = view.linked ? view.linkedUserId : '';
            const extension = createExtensionObject(rigProject.manifest, index.toString(),
              view.viewerType, linkedUserId, view.channelId, rigProject.secret, view.opaqueId);
            return (
              <ExtensionView
                key={`${view.id}:${refreshCount}`}
                view={view}
                extension={extension}
                onOpenEditView={this.openEditViewDialog}
                onDeleteView={onDeleteExtensionView}
                onViewInBrowser={onViewInBrowser}
              />
            );
          })}
        </div>
        <Console isWide={isConsoleWide} onMinimize={this.handleConsoleMinimize} />
      </div>
      {this.state.showingExtensionsViewDialog && (
        <ExtensionViewDialog
          extensionViews={rigProject.manifest.views}
          isBitsEnabled={rigProject.manifest.bitsEnabled}
          closeHandler={this.closeExtensionViewDialog}
          saveHandler={this.createExtensionView}
        />
      )}
      {this.state.viewForEdit && (
        <EditViewDialog
          viewForEdit={this.state.viewForEdit}
          closeHandler={this.closeEditViewDialog}
          updateViewHandler={this.updateView}
        />
      )}
    </>;
  }

  private handleConsoleMinimize = (isConsoleMinimized: boolean) => {
    this.setState({ isConsoleMinimized });
  }
}
