import * as React from 'react';
import './component.sass';
import classNames from 'classnames';
import { Color, CoreText, SVG, SVGAsset, TextType } from 'twitch-core-ui';

interface Props {
  isWide: boolean;
  onMinimize: (isMinimized: boolean) => void;
}

interface ConsoleEntry {
  frameId: string;
  log: string;
}

interface State {
  isMinimized: boolean;
  logHistory: ConsoleEntry[];
}

export class Console extends React.Component<Props, State> {
  private console: React.RefObject<HTMLDivElement>;

  public state: State = {
    isMinimized: false,
    logHistory: Console.constructLogHistory(),
  };

  private static constructLogHistory() {
    return window.rig.history.length ? window.rig.history : [{
      frameId: 'internal',
      log: "Invoke window.Twitch.ext.rig.log('...') in your front-end JavaScript to have messages appear in this console.",
    }];
  }

  constructor(props: Readonly<Props>) {
    super(props);
    this.console = React.createRef();
  }

  private updateConsole = () => {
    setTimeout(() => {
      const console = this.console.current;
      if (console) {
        console.scrollTop = console.scrollHeight;
      }
    });
    this.setState({ logHistory: Console.constructLogHistory() });
  }

  public componentDidMount() {
    window.rig.update = this.updateConsole;
    this.updateConsole();
  }

  private clear = () => {
    window.rig.history.splice(0);
    this.setState({ logHistory: [] });
  }

  public render() {
    const consoleClassName = classNames('console', {
      'console--minimized': this.state.isMinimized,
      'console--wide': this.props.isWide,
    });
    const minimizeTitle = this.state.isMinimized ? 'Restore' : 'Minimize';
    return (
      <div className={consoleClassName} ref={this.console}>
        <div className="console__bar">
          <div title="Clear console log" onClick={this.clear}>
            <SVG asset={SVGAsset.Ban} />
          </div>
          <div title={minimizeTitle}>
            <svg width="16" height="16" viewBox="0 0 16 16" xmlns="http://www.w3.org/2000/svg" onClick={this.toggleMinimize}>
              <rect x="0" y="0" width="16" height="16" strokeWidth={1} />
              {this.state.isMinimized ? (
                <rect x="3" y="4" height="7" width="10" strokeWidth={2} />
              ) : (
                <path d="M 2,9 H 14" strokeWidth={2} />
              )}
            </svg>
          </div>
        </div>
        {this.state.logHistory.map((entry, index) =>
          <CoreText key={index}><CoreText bold color={Color.Link} type={TextType.Span}>{entry.frameId} $</CoreText> {entry.log}</CoreText>)}
      </div>
    );
  }

  private toggleMinimize = () => {
    this.setState((previousState) => {
      const isMinimized = !previousState.isMinimized;
      this.props.onMinimize(isMinimized);
      return { isMinimized };
    });
  }
}
