import { setupMountTest } from '../../tests/enzyme-util/mount';
import { setupShallowTest } from '../../tests/enzyme-util/shallow';
import { ExtensionViewDialog, Props, asFrameSize } from './component';
import { DefaultOverlayPlayerSize } from '../../constants/overlay-sizes'
import { ViewerType } from '../../core/models/rig';
import { ExtensionMode, ExtensionAnchor, ExtensionViews } from 'extension-coordinator';
import { DivOption } from './div-option';
import { timeout, ChangeEventFn } from '../../tests/utils';
import { LocalStorage } from '../../util/local-storage';
import { UserSession } from '../../core/models/user-session';

const localStorage = new LocalStorage();
const userSession: UserSession = {
  authToken: 'authToken',
  displayName: 'displayName',
  email: 'email@email.com',
  expiration: Date.parse('3333-03-03'),
  id: 'id',
  login: 'login',
  profileImageUrl: 'profileImageUrl',
};
localStorage.rigLogin = userSession;

describe('<ExtensionViewDialog />', () => {
  const props = {
    extensionViews: {
      panel: {
        height: 300,
        canLinkExternalContent: false,
        viewerUrl: 'http://test.local',
      },
      liveConfig: {
        canLinkExternalContent: false,
        viewerUrl: 'http://test.local',
      },
      mobile: {
        viewerUrl: 'http://test.local',
      },
      component: {
        canLinkExternalContent: false,
        viewerUrl: 'http://test.local',
      },
      videoOverlay: {
        canLinkExternalContent: false,
        viewerUrl: 'http://test.local',
      },
    } as ExtensionViews,
    isBitsEnabled: false,
    closeHandler: jest.fn(),
    saveHandler: jest.fn(),
  };
  const setupMount = setupMountTest<Props>(ExtensionViewDialog, () => props);
  const setupShallow = setupShallowTest<Props>(ExtensionViewDialog, () => props, { disableLifecycleMethods: true });

  it('focuses on mount', () => {
    const { wrapper } = setupMount();
    const instance = wrapper.instance() as any;
    expect(instance.focusedInput).toBeDefined();
  });

  it('displays mobile options', () => {
    const { wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('DivOption').first().dive().find('input[name="extensionType"]').prop<ChangeEventFn>('onChange')({ currentTarget: { name: 'extensionType', value: 'mobile' } });
    expect(wrapper.find('Radio[name="mobileSize"]')).toHaveLength(8);
    expect(wrapper.find('Radio[name="orientation"]')).toHaveLength(2);
  });

  it('toggles chat enabled', () => {
    const { wrapper } = setupShallow();
    const name = 'isChatEnabled';
    wrapper.find('Dialog').dive().find('CheckBox[label="Is Chat Enabled"]').prop<ChangeEventFn>('onChange')({ currentTarget: { name } });
    expect(wrapper.state('features')).toEqual({ isBitsEnabled: false, [name]: true, isSubscriptionStatusAvailable: false });
  });

  it('toggles popout', () => {
    const { wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('CheckBox[label="Simulate Popout"]').prop<ChangeEventFn>('onChange')({ currentTarget: { name } });
    expect(wrapper.state('isPopout')).toBe(true);
  });

  it('when top nav close button is clicked closeHandler is called', () => {
    const { wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalHeader').dive().find('CoreDismissible').simulate('click');
    const instance = wrapper.instance() as ExtensionViewDialog;
    expect(instance.props.closeHandler).toHaveBeenCalled();
  });

  it('when bottom bar close button is clicked closeHandler is called', () => {
    const { wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive().find('Button').first().simulate('click');
    const instance = wrapper.instance() as ExtensionViewDialog;
    expect(instance.props.closeHandler).toHaveBeenCalled();
  });

  it('when save button is clicked saveHandler is called', () => {
    const { wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive().find('Button').last().simulate('click');
    const instance = wrapper.instance() as ExtensionViewDialog;
    expect(instance.props.saveHandler).toHaveBeenCalled();
  });

  it('reports errors if saveHandler fails due to token', async () => {
    const errorMessage = 'error token';
    const { wrapper } = setupShallow({ saveHandler: jest.fn((_state) => Promise.reject(new Error(errorMessage))) });
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive().find('Button').last().simulate('click');
    await timeout();
    expect(wrapper.state('error')).toBe(errorMessage);
  });

  it('reports errors if saveHandler fails due to non-token', async () => {
    const { wrapper } = setupShallow({ saveHandler: jest.fn((_state) => Promise.reject(new Error('error'))) });
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive().find('Button').last().simulate('click');
    await timeout();
    expect(wrapper.state('error')).toBe('Invalid user id');
  });

  it('only shows the default views if no other views supported', () => {
    const { wrapper } = setupShallow({
      extensionViews: {},
    });

    expect(wrapper.find(DivOption).length).toBe(1);
    expect(wrapper.find(DivOption).someWhere((node: any) => node.prop('value') === ExtensionMode.Config)).toBe(true);
  });

  it('renders opaque id input field correctly', () => {
    const { wrapper } = setupShallow();
    expect(wrapper.find('Dialog').dive().find('Input[name="opaqueId"]')).toHaveLength(1);
  });

  it('changes viewer type', () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    const viewerType = 'viewerType';
    instance.onViewerTypeChange({ currentTarget: { value: viewerType } });
    expect(wrapper.state('identityOption')).toBe('Unlinked');
    expect(wrapper.state('viewerType')).toBe(viewerType);
  });

  it('changes viewer type and identity option', () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    const identityOption = 'identityOption', viewerType = ViewerType.LoggedIn;
    instance.onViewerTypeChange({ currentTarget: { value: `${viewerType}:${identityOption}` } });
    expect(wrapper.state('identityOption')).toBe(identityOption);
    expect(wrapper.state('viewerType')).toBe(viewerType);
  });

  describe('for an extension that supports only video overlays', () => {
    const { wrapper } = setupShallow({
      extensionViews: {
        videoOverlay: {
          canLinkExternalContent: false,
          viewerUrl: 'http://test.local',
        },
      },
    });
    const dive = wrapper.find('Dialog').dive();

    it('renders correctly', () => {
      expect(dive.find(DivOption).length).toBe(2);
      expect(dive.find(DivOption).someWhere((node: any) => node.prop('value') === ExtensionMode.Config)).toBe(true);
      expect(dive.find(DivOption).someWhere((node: any) => node.prop('value') === ExtensionAnchor.Overlay)).toBe(true);
    });

    it('has the correct default state', () => {
      expect(wrapper.state('extensionType')).toBe('video_overlay');
      expect(wrapper.state('frameSize')).toStrictEqual(asFrameSize(DefaultOverlayPlayerSize));
      expect(wrapper.state('viewerType')).toBe(ViewerType.Default);
    });

    it('has the correct selected UI elements', () => {
      expect(dive.find('DivOption[value="video_overlay"][checked=true]')).toHaveLength(1);
      expect(dive.find('DivOption[checked=false]')).toHaveLength(1);
      expect(dive.find('Radio[value="1280x720"][checked=true]')).toHaveLength(1);
      expect(dive.find('Radio[name="overlaySize"][checked=false]')).toHaveLength(4);
    });
  });

  describe('for an extension that supports only panels', () => {
    const { wrapper } = setupShallow({
      extensionViews: {
        panel: {
          height: 300,
          viewerUrl: 'http://test.local',
          canLinkExternalContent: false,
        },
      },
    });
    const dive = wrapper.find('Dialog').dive();

    it('renders correctly', () => {
      expect(dive.find(DivOption).length).toBe(2);
      expect(dive.find(DivOption).someWhere((node: any) => node.prop('value') === ExtensionAnchor.Panel)).toBe(true);
      expect(dive.find(DivOption).someWhere((node: any) => node.prop('value') === ExtensionMode.Config)).toBe(true);
    });

    it('has the correct default state', () => {
      expect(wrapper.state('extensionType')).toBe(ExtensionAnchor.Panel);
      expect(wrapper.state('frameSize')).toStrictEqual(asFrameSize(DefaultOverlayPlayerSize));
      expect(wrapper.state('viewerType')).toBe(ViewerType.Default);
    });

    it('has the correct selected UI elements', () => {
      expect(dive.find('DivOption[value="panel"][checked=true]')).toHaveLength(1);
      expect(dive.find('DivOption[checked=false]')).toHaveLength(1);
    });
  });


  describe('for an extension that only supports components', () => {
    const { wrapper } = setupShallow({
      extensionViews: {
        component: {
          aspectRatioX: 300, 
          aspectRatioY: 200, 
          autoScale: true, 
          canLinkExternalContent: true,
          scalePixels: 1024,
          targetHeight: 9500,
          viewerUrl: 'http://test.local',
          aspectHeight: 0,
          aspectWidth: 0,
          zoom: false,
          zoomPixels: 0,
        },
      },
    });
    const dive = wrapper.find('Dialog').dive();

    it('renders correctly', () => {
      expect(dive.find(DivOption).length).toBe(2);
      expect(dive.find(DivOption).someWhere((node: any) => node.prop('value') === ExtensionAnchor.Component)).toBe(true);
      expect(dive.find(DivOption).someWhere((node: any) => node.prop('value') === ExtensionMode.Config)).toBe(true);
    });

    it('has the correct default state', () => {
      expect(wrapper.state('extensionType')).toBe(ExtensionAnchor.Component);
      expect(wrapper.state('frameSize')).toStrictEqual(asFrameSize(DefaultOverlayPlayerSize));
      expect(wrapper.state('viewerType')).toBe(ViewerType.Default);
    });

    it('has the correct selected UI elements', () => {
      expect(dive.find('DivOption[value="component"][checked=true]')).toHaveLength(1);
      expect(dive.find('DivOption[checked=false]')).toHaveLength(1);
    });
  });
});
