import * as React from 'react';
import './component.sass';
import { OverlayPlayerSizes, DefaultOverlayPlayerSize } from '../../constants/overlay-sizes'
import { ViewerType, DeveloperRigUserId, FrameSize, ExtensionType } from '../../core/models/rig';
import { IdentityOption } from '../../core/models/identity-option';
import { Radio, FormGroup, Layout, Display, Input, InputType, CoreText, TextType, CheckBox, Select, Color, Visibility } from 'twitch-core-ui';
import { DivOption, ExtensionTypeNames } from './div-option';
import { MobileOrientation, MobileSizes, DefaultMobileSize } from '../../constants/mobile';
import { ExtensionAnchor, ExtensionMode, ExtensionPlatform, ExtensionViews, ExtensionInstallationAbilities } from 'extension-coordinator';
import { ChannelIdOrNameInput } from '../../channel-id-or-name-input';
import { Dialog } from '../../dialog';
import { LocalStorage } from '../../util/local-storage';

export interface Props {
  extensionViews: ExtensionViews;
  isBitsEnabled: boolean;
  closeHandler: () => void;
  saveHandler: (definition: ExtensionViewDefinition) => Promise<void>;
}

export interface ExtensionViewDefinition {
  channelId: string;
  configurationAnchor: ExtensionAnchor;
  displayChannelName: string;
  extensionType: ExtensionType;
  features: ExtensionInstallationAbilities;
  frameSize: FrameSize;
  identityOption: IdentityOption;
  isPopout: boolean;
  label: string;
  linkedUserId: string;
  opaqueId?: string;
  orientation: MobileOrientation;
  viewerType: ViewerType;
}

interface State extends ExtensionViewDefinition {
  dashboardAnchor: ExtensionAnchor;
  mobileHeight: string;
  mobileWidth: string;
  overlayHeight: string;
  overlayWidth: string;
  playerHeight: string;
  playerWidth: string;
  error?: string;
  mobileSize: string;
  overlaySize: string;
  playerSize: string;
  supportedAnchors: ExtensionAnchor[];
  [key: string]: any;
}

const localStorage = new LocalStorage();

export function asFrameSize(s: string): FrameSize {
  s = s.split('(').pop() as string;
  const [width, height] = s.split(')')[0].split('x').map(Number);
  return { width, height };
}

function createState(props: Props): State {
  const views = props.extensionViews;
  const supportedAnchors = [
    // Put panel first so it's the default.
    views.panel && views.panel.viewerUrl && ExtensionAnchor.Panel,
    views.videoOverlay && views.videoOverlay.viewerUrl && ExtensionAnchor.Overlay,
    views.component && views.component.viewerUrl && ExtensionAnchor.Component,
  ].filter((type) => type) as ExtensionAnchor[];
  const frameSize = asFrameSize(DefaultOverlayPlayerSize);
  return {
    channelId: localStorage.extensionViewChannelId,
    configurationAnchor: supportedAnchors[0],
    dashboardAnchor: supportedAnchors[0],
    displayChannelName: localStorage.extensionViewDisplayChannelName,
    extensionType: supportedAnchors[0],
    features: {
      isBitsEnabled: props.isBitsEnabled,
      isChatEnabled: false,
      isSubscriptionStatusAvailable: false,
    },
    frameSize,
    identityOption: IdentityOption.Default,
    isPopout: false,
    label: '',
    linkedUserId: DeveloperRigUserId,
    mobileHeight: '',
    mobileSize: DefaultMobileSize,
    mobileWidth: '',
    orientation: MobileOrientation.Default,
    overlayHeight: '',
    overlaySize: DefaultOverlayPlayerSize,
    overlayWidth: '',
    playerHeight: '',
    playerSize: DefaultOverlayPlayerSize,
    playerWidth: '',
    supportedAnchors,
    viewerType: ViewerType.Default,
  };
}

export class ExtensionViewDialog extends React.Component<Props, State> {
  public state: State = createState(this.props);
  private focusedInput?: HTMLInputElement;

  public componentDidMount() {
    this.focusedInput!.focus();
  }

  private onChange = (input: React.FormEvent<HTMLInputElement>) => {
    const { checked, name, type, value } = input.currentTarget;
    switch (name) {
      case 'mobileHeight':
      case 'mobileWidth':
      case 'overlayHeight':
      case 'overlayWidth':
      case 'playerHeight':
      case 'playerWidth':
        if (!/^[0-9]*$/.test(value)) {
          return;
        }
    }
    switch (name) {
      case 'mobileHeight':
      case 'mobileWidth':
        this.setState({ mobileSize: 'Custom' });
        break;
      case 'overlayHeight':
      case 'overlayWidth':
        this.setState({ overlaySize: 'Custom' });
        break;
      case 'playerHeight':
      case 'playerWidth':
        this.setState({ playerSize: 'Custom' });
        break;
    }
    this.setState({ [name]: type === 'checkbox' ? checked : type === 'number' ? Number(value) : value });
  }

  private renderExtensionTypeOptions() {
    const { liveConfig, mobile } = this.props.extensionViews;
    const { extensionType, supportedAnchors } = this.state;
    const supportedTypes = [
      ...supportedAnchors,
      mobile && mobile.viewerUrl && ExtensionPlatform.Mobile,
      ExtensionMode.Config,
      liveConfig && liveConfig.viewerUrl && ExtensionMode.Dashboard,
    ].filter((type) => type) as ExtensionType[];
    const divOptions = supportedTypes.map((type: ExtensionType) => (
      <DivOption
        key={type}
        value={type}
        onChange={this.onChange}
        checked={extensionType === type}
      />
    ));
    return divOptions;
  }

  private close = () => {
    this.props.closeHandler();
  }

  private getFrameSize(): FrameSize {
    const { extensionType, frameSize, mobileHeight, mobileSize, mobileWidth, overlayHeight, overlaySize } = this.state;
    const { overlayWidth, playerHeight, playerSize, playerWidth } = this.state;
    switch (extensionType) {
      case ExtensionAnchor.Component:
        return playerSize === 'Custom' ? { width: Number(playerWidth), height: Number(playerHeight) } : asFrameSize(playerSize);
      case ExtensionAnchor.Overlay:
        return overlaySize === 'Custom' ? { width: Number(overlayWidth), height: Number(overlayHeight) } : asFrameSize(overlaySize);
      case ExtensionPlatform.Mobile:
        return mobileSize === 'Custom' ? { width: Number(mobileWidth), height: Number(mobileHeight) } : asFrameSize(mobileSize);
    }
    return frameSize;
  }

  private save = async () => {
    try {
      const { channelId, configurationAnchor, dashboardAnchor, displayChannelName, extensionType } = this.state;
      const state = {
        ...this.state,
        configurationAnchor: extensionType === ExtensionMode.Dashboard ? dashboardAnchor : configurationAnchor,
        frameSize: this.getFrameSize(),
      };
      await this.props.saveHandler(state);
      localStorage.extensionViewChannelId = channelId;
      localStorage.extensionViewDisplayChannelName = displayChannelName;
    } catch (ex) {
      if (ex.message.includes(' token')) {
        this.setState({ error: ex.message });
      } else {
        const idOrName = ~ex.message.indexOf('Cannot fetch user for login') ? 'name' : 'id';
        this.setState({ error: `Invalid user ${idOrName}` });
      }
    }
  }

  private toggleFeature = (input: React.FormEvent<HTMLInputElement>) => {
    const { name } = input.currentTarget;
    this.setState((previousState) => {
      const features = { ...previousState.features } as { [key: string]: boolean };
      features[name] = !features[name];
      return { features: features as unknown as ExtensionInstallationAbilities };
    });
  }

  private toggleIsPopout = () => {
    this.setState({ isPopout: !this.state.isPopout });
  }

  private onViewerTypeChange = (event: React.FormEvent<HTMLSelectElement>) => {
    const parts = event.currentTarget.value.split(':');
    if (parts[0] === ViewerType.LoggedIn) {
      this.setState({ identityOption: parts[1] as IdentityOption, viewerType: parts[0] });
    } else {
      this.setState({ viewerType: parts[0] as ViewerType });
    }
  }

  private renderViewerTypeSelect = () => {
    const { identityOption, viewerType } = this.state;
    const value = viewerType === ViewerType.LoggedIn ?
      `${viewerType}:${identityOption}` :
      viewerType;
    return <Select value={value} onChange={this.onViewerTypeChange}>
      <option value={ViewerType.Broadcaster}>{ViewerType.Broadcaster}</option>
      <option value={ViewerType.LoggedOut}>{ViewerType.LoggedOut}</option>
      <option value={`${ViewerType.LoggedIn}:${IdentityOption.Unlinked}`}>{ViewerType.LoggedIn}</option>
      <option value={`${ViewerType.LoggedIn}:${IdentityOption.Linked}`}>{ViewerType.LoggedIn} - Shared Identity</option>
    </Select>;
  }

  private renderSizeGroup = (label: string, name: string, sizes: readonly string[]) => {
    const heightName = `${name}Height`;
    const widthName = `${name}Width`;
    const height = this.state[heightName] as string;
    const width = this.state[widthName] as string;
    const sizeName = `${name}Size`;
    const size = this.state[sizeName] as string;
    return <>
      <FormGroup label={label}>
        <div className="extension-view-dialog__group">
          {sizes.map((s) =>
            <Radio key={s} name={sizeName} label={s} value={s} onChange={this.onChange} checked={s === size} />)}
          <Radio name={sizeName} label="Custom" value="Custom" onChange={this.onChange} checked={size === "Custom"} />
          <div className="extension-view-dialog__custom-size">
            <Input error={size === "Custom" && !ExtensionViewDialog.isExtentValid(width)} name={widthName} placeholder="Width"
              type={InputType.Text} value={width} onChange={this.onChange} />
            <CoreText type={TextType.Span}>x</CoreText>
            <Input error={size === "Custom" && !ExtensionViewDialog.isExtentValid(height)} name={heightName} placeholder="Height"
              type={InputType.Text} value={height} onChange={this.onChange} />
          </div>
        </div>
      </FormGroup>
    </>;
  }

  private static isExtentValid(extent: string): boolean {
    const value = parseInt(extent, 10);
    return isFinite(value) && value >= 1;
  }

  private isCustomValid(name: string) {
    const size = this.state[`${name}Size`] as string;
    if (size === 'Custom') {
      const width = this.state[`${name}Width`] as string;
      const height = this.state[`${name}Height`] as string;
      if (!ExtensionViewDialog.isExtentValid(width) || !ExtensionViewDialog.isExtentValid(height)) {
        return false;
      }
    }
    return true;
  }

  private canSave = (): boolean => {
    const { extensionType } = this.state;
    if (extensionType === ExtensionPlatform.Mobile && !this.isCustomValid('mobile')) {
      return false;
    }
    if (extensionType === ExtensionAnchor.Overlay && !this.isCustomValid('overlay')) {
      return false;
    }
    if (extensionType === ExtensionAnchor.Component && !this.isCustomValid('player')) {
      return false;
    }
    return true;
  }

  public render() {
    const { channelId, configurationAnchor, dashboardAnchor, displayChannelName, error, extensionType, features } = this.state;
    const { identityOption, isPopout, linkedUserId, orientation, supportedAnchors, viewerType } = this.state;
    return <>
      <Dialog canConfirm={this.canSave()} className="extension-view-dialog" confirmText="Save" title="Add a New View" closeHandler={this.close} confirmHandler={this.save}>
        <div className="extension-view-dialog__section">
          <FormGroup label="View Type">
            <Layout display={Display.Flex}>
              {this.renderExtensionTypeOptions()}
            </Layout>
          </FormGroup>
          <FormGroup label="View Label">
            <Input name="label" type={InputType.Text} refDelegate={(input) => { this.focusedInput = input; }} onChange={this.onChange} />
          </FormGroup>
          <FormGroup label="Feature Flags">
            <CheckBox checked={features.isBitsEnabled} label="Is Bits Enabled" name="isBitsEnabled" onChange={this.toggleFeature} />
            <CheckBox checked={features.isChatEnabled} label="Is Chat Enabled" name="isChatEnabled" onChange={this.toggleFeature} />
            <CheckBox checked={features.isSubscriptionStatusAvailable} label="Is Subscription Status Available" name="isSubscriptionStatusAvailable" onChange={this.toggleFeature} />
          </FormGroup>
          <FormGroup label="Frame Properties">
            <div className="extension-view-dialog__group">
              <label className="extension-view-dialog__property">
                <CoreText type={TextType.Span}>Channel ID or Name</CoreText>
                <ChannelIdOrNameInput name="channelId" value={channelId} onChange={this.onChange} />
              </label>
              <Layout visibility={extensionType === ExtensionAnchor.Panel || extensionType === ExtensionMode.Dashboard ? Visibility.Visible : Visibility.Hidden}>
                <CheckBox checked={isPopout} label="Simulate Popout" onChange={this.toggleIsPopout} />
              </Layout>
            </div>
          </FormGroup>
          <Layout visibility={extensionType !== ExtensionMode.Config && extensionType !== ExtensionMode.Dashboard ? Visibility.Visible : Visibility.Hidden}>
            <FormGroup label="Viewer Type">
              <div className="extension-view-dialog__group">
                <div className="extension-view-dialog__viewer-types">
                  {this.renderViewerTypeSelect()}
                  {viewerType === ViewerType.LoggedIn && identityOption === IdentityOption.Linked &&
                    <div className="extension-view-dialog__linked-user-id">
                      <ChannelIdOrNameInput name="linkedUserId" value={linkedUserId} placeholder="User ID or Name" onChange={this.onChange} />
                    </div>}
                </div>
                <label className="extension-view-dialog__property">
                  <CoreText type={TextType.Span}>Custom Opaque ID</CoreText>
                  <Input name="opaqueId" type={InputType.Text} onChange={this.onChange} />
                </label>
              </div>
            </FormGroup>
          </Layout>
        </div>
        <div className="extension-view-dialog__vertical-bar" />
        <div className="extension-view-dialog__section">
          {extensionType === ExtensionAnchor.Overlay && this.renderSizeGroup('Overlay Size', 'overlay', OverlayPlayerSizes)}
          {extensionType === ExtensionPlatform.Mobile && this.renderSizeGroup('Screen Size', 'mobile', MobileSizes)}
          {extensionType === ExtensionPlatform.Mobile &&
            <FormGroup label="Orientation">
              <div className="extension-view-dialog__group">
                {[MobileOrientation.Portrait, MobileOrientation.Landscape].map((value) =>
                  <Radio key={value} name="orientation" label={value} value={value} onChange={this.onChange} checked={value === orientation} />)}
              </div>
            </FormGroup>}
          {extensionType === ExtensionAnchor.Component && this.renderSizeGroup('Player Size', 'player', OverlayPlayerSizes)}
          {(extensionType === ExtensionAnchor.Component || extensionType === ExtensionAnchor.Overlay) &&
            <FormGroup label="Display Channel">
              <div className="extension-view-dialog__group">
                <Input name="displayChannelName" type={InputType.Text} value={displayChannelName} onChange={this.onChange} />
              </div>
            </FormGroup>}
          {extensionType === ExtensionMode.Config && supportedAnchors.length > 1 &&
            <FormGroup label="Anchor">
              <div className="extension-view-dialog__group">
                {supportedAnchors.map((key) =>
                  <Radio key={key} name="configurationAnchor" label={ExtensionTypeNames[key]} value={key} onChange={this.onChange} checked={key === configurationAnchor} />)}
              </div>
            </FormGroup>}
          {extensionType === ExtensionMode.Dashboard && supportedAnchors.length > 1 &&
            <FormGroup label="Anchor">
              <div className="extension-view-dialog__group">
                {supportedAnchors.map((key) =>
                  <Radio key={key} name="dashboardAnchor" label={ExtensionTypeNames[key]} value={key} onChange={this.onChange} checked={key === dashboardAnchor} />)}
              </div>
            </FormGroup>}
          <CoreText color={Color.Error}>{error}</CoreText>
        </div>
      </Dialog>
    </>;
  }
}
