import { setupShallowTest } from '../../tests/enzyme-util/shallow';
import { ExtensionView } from './component';
import { ViewerType } from '../../core/models/rig';
import { createExtensionObjectForTest, createViewsForTest } from '../../tests/constants/extension';
import { ExtensionAnchor, ExtensionMode, ExtensionPlatform, ExtensionObject } from 'extension-coordinator';
import { MobileOrientation } from '../../constants/mobile';

const testExtension = createExtensionObjectForTest();
describe('<ExtensionView />', () => {
  const setupShallow = setupShallowTest(ExtensionView, () => ({
    extension: testExtension,
    view: createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.LoggedOut)[0],
    onDeleteView: jest.fn(),
    onOpenEditView: jest.fn(),
    onViewInBrowser: jest.fn(),
  }));

  it('invokes onDeleteView upon clicking the delete button', () => {
    const { props, wrapper } = setupShallow();
    expect(wrapper.find('Button')).toHaveLength(5);
    wrapper.find('Button').last().simulate('click');
    expect(props.onDeleteView).toHaveBeenCalled();
  });

  it('invokes onViewInBrowser upon clicking the "View in browser" button', () => {
    const { props, wrapper } = setupShallow();
    wrapper.find('Button').at(3).simulate('click');
    expect(props.onViewInBrowser).toHaveBeenCalled();
  });

  it('refreshes a view', () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    instance.iframe = {};
    wrapper.find('Button').at(2).simulate('click');
    expect(wrapper.state('frameId')).toBe(2);
  });

  it('selects an item', () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    const expectedValue = 'test';
    instance.iframe = { contentWindow: { postMessage: jest.fn() } };
    instance.selectItem({ currentTarget: { innerText: expectedValue } });
    expect(wrapper.state('selectedContextMenuItem')).toBe(expectedValue);
  });

  it('selects a sub-item', () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    instance.iframe = { contentWindow: { postMessage: jest.fn() } };
    instance.selectSubitem({ currentTarget: { name: 'volume', value: '1' } });
    expect(instance.iframe.contentWindow.postMessage).toHaveBeenCalled();
  });

  it('shows and hides the "Edit Context" menu', () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    instance.toggleContextMenu();
    expect(wrapper.state('showingContextMenu')).toBeTruthy();
    instance.onClick({ srcElement: {} })
    expect(wrapper.state('showingContextMenu')).toBeFalsy();
  });

  describe('component', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionAnchor.Component, ViewerType.LoggedOut, { x: 10, y: 10 })[0],
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('ComponentView').length).toBe(1);
    });
  });

  describe('config', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionMode.Config, ViewerType.Broadcaster)[0],
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });
  });

  describe('live config', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionMode.Dashboard, ViewerType.Broadcaster)[0],
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('sets correct live config height when panel height provided', () => {
      const extensionWithPanelHeight: ExtensionObject = {
        ...createExtensionObjectForTest(),
        views: {
          config: {
            canLinkExternalContent: false,
            viewerUrl: 'http://test.local',
          },
          liveConfig: {
            canLinkExternalContent: false,
            viewerUrl: 'http://test.local',
          },
          panel: {
            canLinkExternalContent: false,
            viewerUrl: 'http://test.local',
            height: 500,
          }
        },
      };

      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionMode.Dashboard, ViewerType.Broadcaster)[0],
        extension: extensionWithPanelHeight
      });

      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(extensionWithPanelHeight);
    });
  });

  describe('mobile', () => {
    it('renders correctly', () => {
      const extras = { orientation: MobileOrientation.Portrait };
      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionPlatform.Mobile, ViewerType.LoggedOut, extras)[0],
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('MobileView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });
  });

  describe('overlay', () => {
    it('renders correctly as a Broadcaster', () => {
      const view = createViewsForTest(1, ExtensionAnchor.Overlay, ViewerType.Broadcaster)[0];
      view.frameSize = {
        height: 1,
        width: 1
      };
      const { wrapper } = setupShallow({
        view,
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('OverlayView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('renders correctly as a Logged In and Linked viewer', () => {
      const view = createViewsForTest(1, ExtensionAnchor.Overlay, ViewerType.LoggedIn, { x: 10, y: 10 })[0];
      view.linked = true;
      view.frameSize = {
        height: 1,
        width: 1
      };
      const { wrapper } = setupShallow({
        view,
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('OverlayView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('renders correctly as a Logged In and Unlinked viewer', () => {
      const view = createViewsForTest(1, ExtensionAnchor.Overlay, ViewerType.LoggedIn)[0];
      view.frameSize = {
        height: 1,
        width: 1
      };
      const { wrapper } = setupShallow({
        view,
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('OverlayView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('renders correctly as a Logged Out viewer', () => {
      const view = createViewsForTest(1, ExtensionAnchor.Overlay, ViewerType.LoggedOut, { x: 10, y: 10 })[0];
      view.frameSize = {
        height: 1,
        width: 1
      };
      const { wrapper } = setupShallow({
        view,
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('OverlayView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });
  });

  describe('panel', () => {
    it('renders correctly as a Broadcaster', () => {
      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.Broadcaster)[0],
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('renders correctly as a Logged In and Linked viewer', () => {
      const view = createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.LoggedIn, { x: 10, y: 10 })[0];
      view.linked = true;
      const { wrapper } = setupShallow({
        view,
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('renders correctly as a Logged In and Unlinked viewer', () => {
      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.LoggedIn)[0],
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('renders correctly as a Logged Out viewer', () => {
      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.LoggedOut)[0],
      });
      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(testExtension);
    });

    it('sets correct panel height when panel height provided', () => {
      const extensionWithPanelHeight: ExtensionObject = {
        ...createExtensionObjectForTest(),
        views: {
          config: {
            canLinkExternalContent: false,
            viewerUrl: 'http://test.local',
          },
          liveConfig: {
            canLinkExternalContent: false,
            viewerUrl: 'http://test.local',
          },
          panel: {
            canLinkExternalContent: false,
            viewerUrl: 'http://test.local',
            height: 400,
          }
        },
      };

      const { wrapper } = setupShallow({
        view: createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.LoggedOut)[0],
        extension: extensionWithPanelHeight,
      });

      expect(wrapper.debug()).toMatchSnapshot();
      expect(wrapper.find('ExtensionViewSelector').dive().find('StandardView').dive().find('ExtensionFrame').prop('extension')).toBe(extensionWithPanelHeight);
    });
  });
});
