import * as React from 'react';
import './component.sass';
import { IdentityOption } from '../../core/models/identity-option';
import { RigExtensionView, ViewerType, ExtensionViewTypeKeys } from '../../core/models/rig';
import { ExtensionAnchor, ExtensionPlatform, ExtensionObject } from 'extension-coordinator';
import { BalloonDirection, Button, ButtonType, DropDownMenu, Layout, Position, DropDownMenuInputItemType, SVGAsset, DropDownMenuItemProps, BalloonSize, Balloon, Input, InputType, SVG, CoreText, StyledLayout, AlignItems, Background, Display, FontSize, ButtonSize } from 'twitch-core-ui';
import electron from '../../electron';
import { sendEvent } from '../../util/api';
import { parse } from 'url';
import classNames from 'classnames';
import { ExtensionViewSelector } from '../../extension-view-selector';

interface Props {
  extension: ExtensionObject;
  view: RigExtensionView;
  onDeleteView: (id: string) => void;
  onOpenEditView: (id: string) => Promise<void>;
  onViewInBrowser: (id: string) => void;
}

interface Context {
  [key: string]: boolean | string;
}

interface State {
  context: Context;
  isExtensionVisible: boolean;
  isNonoVisible: boolean;
  isViewVisible: boolean;
  showingContextMenu: boolean;
  showingEditDialog: boolean;
  selectedContextMenuItem?: string;
  frameId: number;
}

const resolutions = ['480x270', '640x360', '800x450', '1280x720', '1600x900', '1920x1080'];
const defaultResolution = resolutions[3];

const defaultContext: Context = {
  arePlayerControlsVisible: false,
  bitrate: '5000',
  bufferSize: '100',
  displayResolution: defaultResolution,
  game: "Don't Starve",
  hlsLatencyBroadcaster: '5',
  isFullScreen: false,
  isMuted: false,
  isPaused: false,
  isTheatreMode: false,
  language: 'en',
  playbackMode: 'video',
  theme: 'light',
  videoResolution: defaultResolution,
  volume: '.5',
};

const menuItems: { [key: string]: string[] | number | string | undefined; } = {
  arePlayerControlsVisible: undefined,
  displayResolution: resolutions,
  game: '-',
  hlsLatencyBroadcaster: 60,
  isFullScreen: undefined,
  isMuted: undefined,
  isPaused: undefined,
  isTheatreMode: undefined,
  language: ['en', 'da', 'de', 'en-gb', 'es', 'es-mx', 'fr', 'it', 'hu', 'nl', 'no', 'pl', 'pt', 'pt-br', 'sk', 'fi', 'sv', 'vi', 'tr', 'cs', 'el', 'bg', 'ru'],
  playbackMode: ['video', 'audio', 'remote', 'chat-only'],
  theme: ['light', 'dark'],
  videoResolution: resolutions,
  volume: 1,
};

const numericItems = ['bitrate', 'bufferSize', 'hlsLatencyBroadcaster', 'volume'];

export class ExtensionView extends React.Component<Props, State> {
  state: State = {
    context: {
      ...defaultContext,
    },
    isExtensionVisible: true,
    isNonoVisible: true,
    isViewVisible: true,
    showingContextMenu: false,
    showingEditDialog: false,
    frameId: 1,
  };
  private contextMenuParent?: HTMLDivElement;
  private iframe?: HTMLIFrameElement;
  private items: DropDownMenuItemProps[];
  private lastContext: Context = {};
  private names: string[] = Object.keys(menuItems).sort();

  constructor(props: Readonly<Props>) {
    super(props);
    this.items = this.names.map((title) => ({ title, figure: { icon: SVGAsset.GlyphArrRight }, figureRight: true, onClick: this.selectItem }));
    this.items.push({ title: 'Learn more about Context', linkTo: 'https://dev.twitch.tv/docs/extensions/reference/#oncontext', targetBlank: true });
  }

  componentDidMount() {
    document.addEventListener('click', this.onClick);
  }

  componentWillUnmount() {
    document.removeEventListener('click', this.onClick);
  }

  private onClick = (event: MouseEvent) => {
    if (this.state.showingContextMenu) {
      for (let element = event.srcElement as Element | null; element; element = element.parentElement) {
        if (element === this.contextMenuParent) {
          return;
        }
      }
      this.toggleContextMenu();
    }
  }

  private setContextMenuParent = (element: HTMLDivElement) => {
    this.contextMenuParent = element;
  }

  private setIframe = (iframe: HTMLIFrameElement) => {
    this.iframe = iframe;
  }

  private refreshView = () => {
    this.setState((previousState) => ({ frameId: previousState.frameId + 1 }));
  }

  private selectItem = (event: React.FormEvent<HTMLElement>) => {
    const selectedContextMenuItem = event.currentTarget.innerText.trim();
    this.setState({ selectedContextMenuItem });
  }

  private selectSubitem = (event: React.FormEvent<HTMLInputElement>) => {
    const { name: nameId, value } = event.currentTarget;
    const fieldName = nameId.split('-')[0];
    const menuItem = menuItems[fieldName];
    const fieldValue = menuItem ? value : value === 'on';
    if (typeof menuItem !== 'number' || this.isValidNumeric(fieldName, fieldValue)) {
      const extensionFieldValue = typeof menuItem === 'number' ? Number(value) : fieldValue;
      const context = this.createExtensionContext(fieldName, extensionFieldValue);
      if (JSON.stringify(context) !== JSON.stringify(this.lastContext)) {
        const message = {
          action: 'extension-frame-context',
          context,
          updatedFields: [fieldName],
        };
        this.iframe!.contentWindow!.postMessage(message, '*');
        const eventData = {
          view_type: this.props.view.type,
          context_name: fieldName,
          original_value: this.lastContext[fieldName],
          final_value: fieldValue,
        };
        sendEvent('dx_rig_edit_extension_context', eventData);
        this.lastContext = context;
      }
    }
    this.setState((previousValue) => ({ context: { ...previousValue.context, [fieldName]: fieldValue } }));
  }

  private constructDescription = (): string => {
    const { extension: { views }, view } = this.props;
    const parts: string[] = [view.viewerType];
    if (view.viewerType === ViewerType.LoggedIn) {
      parts.unshift(view.linked ? IdentityOption.Linked : IdentityOption.Unlinked);
    }
    parts.push(view.type.replace(/.*_/, ''));
    if (view.isPopout) {
      parts.push('Popout');
    }
    try {
      const { pathname } = parse((views as any)[ExtensionViewTypeKeys[view.type]].viewerUrl);
      parts.push(`(${electron.basename(pathname)})`);
    } catch (ex) {
      parts.push('(INVALID)');
    }
    if (view.label) {
      parts.push(view.label);
    }
    return parts.join(' ');
  }

  private toggleContextMenu = () => {
    this.setState((previousState) => ({ selectedContextMenuItem: undefined, showingContextMenu: !previousState.showingContextMenu }));
  }

  private createExtensionContext(fieldName: string, fieldValue: boolean | number | string): {} {
    const { context } = this.state;
    return {
      ...context,
      ...numericItems.reduce((items, name) => {
        items[name] = Number(context[name]);
        return items;
      }, {} as { [key: string]: number; }),
      [fieldName]: fieldValue,
    };
  }

  private isValidNumeric(name: string, fieldValue?: boolean | string): boolean {
    const value = Number(fieldValue || this.state.context[name]);
    const maximumValue = menuItems[name] as number;
    return !isNaN(value) && value >= 0 && value <= maximumValue;
  }

  private renderSubmenuItem = (name: string): JSX.Element => {
    const { context, selectedContextMenuItem } = this.state;
    const { view: { id } } = this.props;
    const nameId = `${name}-${id}`;
    const menuItem = menuItems[name];
    if (typeof menuItem === 'number' || typeof menuItem === 'string') {
      return (
        <Balloon direction={BalloonDirection.Right} show={selectedContextMenuItem === name}>
          <Input error={typeof menuItem === 'number' && !this.isValidNumeric(name)} name={nameId} type={InputType.Text} value={context[name] as string} onChange={this.selectSubitem} />
        </Balloon>
      );
    } else {
      const childItems = Array.isArray(menuItem) ? menuItem.map((value) => ({
        type: DropDownMenuInputItemType.Radio, name: nameId, label: value, value, defaultChecked: context[name] === value, onChange: this.selectSubitem,
      })) : [
        { type: DropDownMenuInputItemType.Radio, name: nameId, label: 'Yes', value: 'on', onChange: this.selectSubitem },
        { type: DropDownMenuInputItemType.Radio, name: nameId, label: 'No', value: 'off', defaultChecked: true, onChange: this.selectSubitem },
      ];
      return (
        <DropDownMenu direction={BalloonDirection.Right as unknown as undefined}
          show={selectedContextMenuItem === name} items={childItems} />
      );
    }
  }

  private openEditView = async (id: string) => {
    this.setState({ showingEditDialog: true });
    await this.props.onOpenEditView(id);
    this.setState({ showingEditDialog: false });
  }

  private toggleExtensionVisibility = () => {
    this.setState((previousState) => ({ isExtensionVisible: !previousState.isExtensionVisible }));
  }

  private toggleNonoVisibility = () => {
    this.setState((previousState) => ({ isNonoVisible: !previousState.isNonoVisible }));
  }

  private toggleViewVisibility = () => {
    this.setState((previousState) => ({ isViewVisible: !previousState.isViewVisible }));
  }

  private renderContextMenu = () => {
    const { view: { id, type } } = this.props;
    const { isExtensionVisible, isNonoVisible, showingEditDialog } = this.state;
    const nonoToggleText = `${isNonoVisible ? 'Hide' : 'Show'} no-no zone`;
    const overlayToggleText = `${isExtensionVisible ? 'Hide' : 'Show'} overlay`;
    return <>
      <div className="extension-view__context-menu-handle">
        <svg width="28" height="28" viewBox="0 0 28 28" fill="none" xmlns="http://www.w3.org/2000/svg">
          <path fill="#6441A4" d="M15.9998 7.9998C15.9998 9.1038 15.1038 9.9998 13.9998 9.9998C12.8958 9.9998 11.9998 9.1038 11.9998 7.9998C11.9998 6.8958 12.8958 5.9998 13.9998 5.9998C15.1038 5.9998 15.9998 6.8958 15.9998 7.9998ZM15.9998 13.9998C15.9998 15.1038 15.1038 15.9998 13.9998 15.9998C12.8958 15.9998 11.9998 15.1038 11.9998 13.9998C11.9998 12.8958 12.8958 11.9998 13.9998 11.9998C15.1038 11.9998 15.9998 12.8958 15.9998 13.9998ZM13.9998 21.9998C15.1038 21.9998 15.9998 21.1038 15.9998 19.9998C15.9998 18.8958 15.1038 17.9998 13.9998 17.9998C12.8958 17.9998 11.9998 18.8958 11.9998 19.9998C11.9998 21.1038 12.8958 21.9998 13.9998 21.9998Z" />
          <rect className={`extension-view__context-menu-${showingEditDialog ? 'hidden' : 'rect'}`} stroke="#6441A4" x="0.5" y="0.5" width="27" height="27" rx="2.5" />
        </svg>
      </div>
      <Layout className={`extension-view__context-menu-${showingEditDialog ? 'hidden' : 'content'}`}>
        <Balloon direction={BalloonDirection.BottomRight} show>
          <StyledLayout background={Background.Base} padding={0.5}>
            {type === ExtensionPlatform.Mobile &&
              <Button ariaLabel="ExtensionView:Edit Orientation" fullWidth icon={SVGAsset.Edit} type={ButtonType.Text} onClick={() => this.openEditView(id)}>
                <div className="extension-view__context-menu-text"><CoreText fontSize={FontSize.Size5}>Edit Orientation</CoreText></div>
              </Button>}
            {(type === ExtensionAnchor.Component || type === ExtensionAnchor.Overlay) &&
              <Button ariaLabel={`ExtensionView:${nonoToggleText}`} fullWidth icon={isNonoVisible ? SVGAsset.VisibilityHidden : SVGAsset.VisibilityShown} type={ButtonType.Text} onClick={this.toggleNonoVisibility}>
                <div className="extension-view__context-menu-text"><CoreText fontSize={FontSize.Size5}>{nonoToggleText}</CoreText></div>
              </Button>}
            {type === ExtensionAnchor.Overlay &&
              <Button ariaLabel={`ExtensionView:${overlayToggleText}`} fullWidth icon={isExtensionVisible ? SVGAsset.VisibilityHidden : SVGAsset.VisibilityShown} type={ButtonType.Text} onClick={() => this.toggleExtensionVisibility()}>
                <div className="extension-view__context-menu-text"><CoreText fontSize={FontSize.Size5}>{overlayToggleText}</CoreText></div>
              </Button>}
            <Button ariaLabel="ExtensionView:View in browser" fullWidth icon={SVGAsset.Link} type={ButtonType.Text} onClick={() => this.props.onViewInBrowser(id)}>
              <div className="extension-view__context-menu-text"><CoreText fontSize={FontSize.Size5}>View in browser</CoreText></div>
            </Button>
            <Button ariaLabel="ExtensionView:Delete View" fullWidth icon={SVGAsset.Trash} type={ButtonType.Text} onClick={() => this.props.onDeleteView(id)}>
              <div className="extension-view__context-menu-text"><CoreText fontSize={FontSize.Size5}>Delete View</CoreText></div>
            </Button>
          </StyledLayout>
        </Balloon>
      </Layout>
    </>;
  }

  public render() {
    const { view } = this.props;
    const { isExtensionVisible, isNonoVisible, isViewVisible, showingContextMenu } = this.state;
    const viewClassNames = classNames('extension-view', { 'extension-view--hidden': !isViewVisible });
    const horizontalOffset = '4em';
    const submenus = this.names.map((name, index) => (
      <div key={name} style={{ position: 'relative', right: horizontalOffset, top: `${18 + 30 * index}px` }}>
        {this.renderSubmenuItem(name)}
      </div>
    ));
    return (
      <StyledLayout className={viewClassNames} background={Background.Alt} border flexGrow={1} margin={1}>
        <StyledLayout alignItems={AlignItems.Center} background={Background.Base} borderBottom display={Display.Flex} padding={{ bottom: 0.5, left: 1, top: 0.5 }}>
          <Button ariaLabel={`ExtensionView:${isViewVisible ? 'Collapse' : 'Expand'}View`} type={ButtonType.Text} onClick={this.toggleViewVisibility}><CoreText bold>{this.constructDescription()}</CoreText></Button>
          <Layout margin={{ left: 'auto', right: 1 }} refDelegate={this.setContextMenuParent} position={Position.Relative}>
            <Button ariaLabel="ExtensionView:Edit Context" dropdown type={ButtonType.Hollow} onClick={this.toggleContextMenu}>Edit Context</Button>
            <DropDownMenu offsetX={horizontalOffset} size={BalloonSize.Small} direction={BalloonDirection.BottomRight} show={showingContextMenu} items={this.items} />
            {submenus}
          </Layout>
          <div className="extension-view__button">
            <Button ariaLabel="ExtensionView:Refresh View" title="Refresh View" type={ButtonType.Text} onClick={this.refreshView}>
              <SVG asset={SVGAsset.Refresh} />
            </Button>
          </div>
          <StyledLayout className="extension-view__context-menu">
            {isViewVisible ? this.renderContextMenu() : <Layout alignItems={AlignItems.Center} display={Display.Flex} fullHeight fullWidth>
              <Button ariaLabel="ExtensionView:Expand View" icon={SVGAsset.Fullscreen} size={ButtonSize.Small} title="Expand View" onClick={this.toggleViewVisibility} />
            </Layout>}
          </StyledLayout>
        </StyledLayout>
        <ExtensionViewSelector key={`${view.id}:${this.state.frameId}`} extension={this.props.extension}
          hasBorder isExtensionVisible={isExtensionVisible} isNonoVisible={isNonoVisible} view={view} onDeleteView={() => this.props.onDeleteView(view.id)} onIframeChanged={this.setIframe} />
      </StyledLayout>
    );
  }
}
