import * as React from 'react';
import './component.sass';
import { CloseButton } from './components/close-button';
import { SettingsButton } from './components/settings-button';
import { ExtensionFrame, Props as FrameProps } from '../extension-frame';
import { FrameSize } from '../../core/models/rig';
import classNames from 'classnames';

export interface Props extends FrameProps {
  displayChannelName: string;
  frameSize: FrameSize;
  isNonoVisible: boolean;
}

interface State {
  frameKey: number;
  isHidden: boolean;
}

const SIDEBAR_EDGE_DISTANCE = 75; // the distance between the right edge of the player and the left edge of the sidebar
const TOP_NONO_ZONE_HEIGHT = 100;
const BOTTOM_NONO_ZONE_HEIGHT = 80;
const TOP_BAR_HEIGHT = 30; // the height of the navbar on top of component extensions
const USED_HEIGHT = TOP_BAR_HEIGHT + TOP_NONO_ZONE_HEIGHT + BOTTOM_NONO_ZONE_HEIGHT;

export class ComponentView extends React.Component<Props, State> {
  public state: State = {
    frameKey: 0,
    isHidden: false,
  };

  private computeComponentDimensions() {
    const component = this.props.extension.views.component!;
    const targetHeight = (component.targetHeight * (this.props.frameSize.height - USED_HEIGHT)) / 10000;
    const targetWidth = (targetHeight * component.aspectRatioX) / component.aspectRatioY;
    const targetZoom = component.autoScale ? targetWidth / component.scalePixels : 1;
    return {
      height: targetHeight,
      width: targetWidth,
      zoomScale: targetZoom,
    };
  }

  private computeViewStyles(): React.CSSProperties {
    const playerHeight = this.props.frameSize.height;
    const { height, width, zoomScale } = this.computeComponentDimensions();
    // the center point between the edges of the no-no zones
    const extensionCenter = (playerHeight + TOP_NONO_ZONE_HEIGHT - BOTTOM_NONO_ZONE_HEIGHT) / 2;
    return {
      border: '1px solid #7D55C7',
      right: SIDEBAR_EDGE_DISTANCE + 'px',
      position: 'absolute',
      top: extensionCenter - (height - TOP_BAR_HEIGHT) / 2 + 'px',
      visibility: this.state.isHidden ? 'hidden' : 'visible',
      transform: zoomScale !== 1 ? `scale(${zoomScale})` : 'none',
      transformOrigin: '100% 0',
      height: `${height / zoomScale}px`,
      width: `${width / zoomScale}px`,
    }
  }

  private computeHeaderStyles(top: number | string | undefined): React.CSSProperties {
    const { width } = this.computeComponentDimensions();
    return {
      right: SIDEBAR_EDGE_DISTANCE + 'px',
      top,
      marginTop: -TOP_BAR_HEIGHT + 'px',
      visibility: this.state.isHidden ? 'hidden' : 'visible',
      width: width + 'px',
    };
  }

  private handleSettingsClick = (_event: React.MouseEvent<HTMLDivElement, MouseEvent>) => {
    this.setState((previousState) => ({ frameKey: previousState.frameKey + 1 }));
  }

  public render() {
    const { displayChannelName, extension: { name, iconUrls: { square24: extensionIconUrl } }, frameSize, isNonoVisible } = this.props;
    const viewClassNames = classNames('view', { 'nono-zone': isNonoVisible });
    const viewStyles = this.computeViewStyles();
    const playerUrl = `/player?channel=${displayChannelName}&height=${frameSize.height}`;
    return (
      <div
        className={viewClassNames}
        style={{
          height: `${frameSize.height}px`,
          width: `${frameSize.width}px`,
        }}>
        {displayChannelName &&
          <webview src={playerUrl} style={{ height: `${frameSize.height + 4}px` }} title={displayChannelName} />}
        <div className="component-header" style={this.computeHeaderStyles(viewStyles.top)}>
          <div className="component-header-text">
            {name}
          </div>
          <SettingsButton onClick={this.handleSettingsClick} />
          <CloseButton onClick={this.toggleHide} />
        </div>
        <div className="component-frame-background" style={viewStyles}>
          <ExtensionFrame {...this.props} hidden={this.state.isHidden} key={this.state.frameKey} />
        </div>
        <div className="component-sidebar">
          <div className="component-sidebar-card-container">
            <div className="component-sidebar-card" />
            <div className="component-sidebar-card component-sidebar-card-main" onClick={this.toggleHide}>
              {extensionIconUrl && <>
                <img alt="+" className="component-sidebar-card-image" src={extensionIconUrl} />
              </>}
            </div>
            <div className="component-sidebar-card" />
          </div>
        </div>
      </div>
    );
  }

  private toggleHide = () => {
    this.setState({ isHidden: !this.state.isHidden });
  }
}
