import * as React from 'react';
import './component.sass';
import { createExtensionFrameProps } from './extension-frame';
import { ComponentView } from './component-view';
import { MobileView } from './mobile-view';
import { RigExtensionView, ExtensionViewTypeKeys } from '../core/models/rig';
import { ExtensionAnchor, ExtensionPlatform, ExtensionObject } from 'extension-coordinator';
import { OverlayView } from './overlay-view';
import { StandardView } from './standard-view';
import { Layout, AlignItems, Display, FlexDirection, Button, SVGAsset, CoreText, TextType } from 'twitch-core-ui';

interface Props {
  extension: ExtensionObject;
  hasBorder: boolean;
  isExtensionVisible: boolean;
  isNonoVisible: boolean;
  view: RigExtensionView;
  onDeleteView: () => void;
  onIframeChanged?: (iframe: HTMLIFrameElement) => void;
}

export class ExtensionViewSelector extends React.Component<Props> {
  private bindIframeToParent = (iframe: HTMLIFrameElement) => {
    if (iframe) {
      const coordinatorScriptElement = document.getElementById('coordinatorScriptElement') as HTMLScriptElement;
      const attribute = iframe.contentDocument!.createAttribute('coordinatorUrl');
      attribute.value = coordinatorScriptElement.src;
      iframe.attributes.setNamedItem(attribute);
    }
    const { onIframeChanged } = this.props;
    if (onIframeChanged) {
      onIframeChanged(iframe);
    }
  }

  public render() {
    const { extension, hasBorder, isExtensionVisible, isNonoVisible, view, onDeleteView } = this.props;
    const { displayChannelName, frameSize, orientation, type } = view;
    const extensionFrameProps = createExtensionFrameProps(extension, view, this.bindIframeToParent);
    if (extension.views[ExtensionViewTypeKeys[type]] === undefined) {
      return <Layout alignItems={AlignItems.Center} display={Display.Flex} flexDirection={FlexDirection.Column} margin={4}>
        <CoreText type={TextType.H4}>Due to a change in the Extension manifest, this view is no longer available.</CoreText>
        <Button ariaLabel="ExtensionViewSelector:Delete View" icon={SVGAsset.Trash} onClick={onDeleteView}>Delete View</Button>
      </Layout>
    }
    switch (type) {
      case ExtensionAnchor.Component:
        return (
          <ComponentView
            {...extensionFrameProps}
            displayChannelName={displayChannelName}
            frameSize={frameSize}
            isNonoVisible={isNonoVisible}
          />
        );
      case ExtensionPlatform.Mobile:
        return (
          <MobileView
            {...extensionFrameProps}
            frameSize={frameSize}
            orientation={orientation!}
          />
        );
      case ExtensionAnchor.Overlay:
        return (
          <OverlayView
            {...extensionFrameProps}
            displayChannelName={displayChannelName}
            frameSize={frameSize}
            isExtensionVisible={isExtensionVisible}
            isNonoVisible={isNonoVisible}
          />
        );
      default:
        return (
          <StandardView {...extensionFrameProps} hasBorder={hasBorder} panelHeight={extension.views.panel && extension.views.panel.height} type={type} />
        );
    }
  }
}
