import { setupShallowTest } from '../../tests/enzyme-util/shallow';
import { setupMountTest } from '../../tests/enzyme-util/mount';
import { createExtensionObjectForTest } from '../../tests/constants/extension';
import { ExtensionFrame, Props } from '.';
import { ExtensionAnchor, ExtensionMode, ExtensionPlatform } from 'extension-coordinator';

const port = 9999;
function mockApiFunctions() {
  const original = require.requireActual('../../util/api');
  return {
    ...original,
    fetchHostingStatus: jest.fn(() => Promise.resolve({ port })),
  };
}
jest.mock('../../util/api', () => mockApiFunctions());
const api = require.requireMock('../../util/api');
const channelId = 'channelId';
const defaultPropGenerator = () => ({
  anchor: ExtensionAnchor.Panel,
  channelId,
  extension: createExtensionObjectForTest(port),
  frameId: '0',
  installationAbilities: {
    isBitsEnabled: true,
    isChatEnabled: true,
    isSubscriptionStatusAvailable: false,
  },
  isPopout: false,
  mode: ExtensionMode.Viewer,
  platform: ExtensionPlatform.Web,
  bindIframeToParent: jest.fn(),
});

const setupShallow = setupShallowTest<Props>(ExtensionFrame, defaultPropGenerator);
const setupMount = setupMountTest<Props>(ExtensionFrame, defaultPropGenerator);

describe('<ExtensionFrame />', () => {
  it('invokes postMessage correctly', async () => {
    const { wrapper } = setupMount();
    const mockIframeRef: any = {
      contentWindow: {
        postMessage: jest.fn(),
      },
    };
    const instance = wrapper.instance() as any;
    instance.iframe = mockIframeRef;
    await instance.extensionFrameInit();
    expect(mockIframeRef.contentWindow.postMessage).toHaveBeenCalledWith(createExpected('panel'), '*');
    expect(api.fetchHostingStatus).toHaveBeenCalled();
  });

  it('invokes postMessage correctly when platform is mobile', async () => {
    const { wrapper } = setupMount({
      anchor: ExtensionAnchor.Panel,
      platform: ExtensionPlatform.Mobile,
    });
    const mockIframeRef: any = {
      contentWindow: {
        postMessage: jest.fn(),
      },
    };
    const instance = wrapper.instance() as any;
    instance.iframe = mockIframeRef;
    await instance.extensionFrameInit();
    expect(mockIframeRef.contentWindow.postMessage).toHaveBeenCalledWith(createExpected('panel', true), '*');
  });

  it('invokes postMessage when the visibility changes', async () => {
    const { wrapper } = setupMount();
    const mockIframeRef: any = {
      contentWindow: {
        postMessage: jest.fn(),
      },
    };

    const instance = wrapper.instance() as any;
    instance.iframe = mockIframeRef;
    await instance.extensionFrameInit();

    wrapper.setProps({ hidden: true });
    expect(mockIframeRef.contentWindow.postMessage).toHaveBeenCalledWith({
      action: 'extension-frame-visibility',
      isVisible: false,
    }, '*');

    mockIframeRef.contentWindow.postMessage.mockClear();

    wrapper.setProps({ hidden: false });
    expect(mockIframeRef.contentWindow.postMessage).toHaveBeenCalledWith({
      action: 'extension-frame-visibility',
      isVisible: true,
    }, '*');
  });

  describe('dashboard', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow({
        mode: ExtensionMode.Dashboard,
      });
      expect(wrapper.debug()).toMatchSnapshot();
    });
  });

  describe('config', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow({
        mode: ExtensionMode.Config,
      });
      expect(wrapper.debug()).toMatchSnapshot();
    });
  });

  describe('panel', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow();
      expect(wrapper.debug()).toMatchSnapshot();
    });
  });

  describe('video overlay', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow({
        anchor: ExtensionAnchor.Overlay,
      });
      expect(wrapper.debug()).toMatchSnapshot();
    });
  });

  describe('component', () => {
    it('renders correctly', () => {
      const { wrapper } = setupShallow({
        anchor: ExtensionAnchor.Component,
      });
      expect(wrapper.debug()).toMatchSnapshot();
    });
  });

  function createExpected(anchor: string, mobile = false) {
    const expected = {
      action: 'extension-frame-init',
      channelId,
      parameters: {
        anchor,
        channelId: NaN,
        configuration: undefined,
        extension: createExtensionObjectForTest(port),
        iframeClassName: 'extension-frame',
        installationAbilities: {
          isBitsEnabled: true,
          isChatEnabled: true,
          isSubscriptionStatusAvailable: false,
        },
        isPopout: false,
        loginId: null,
        mode: ExtensionMode.Viewer,
        platform: mobile ? 'mobile' : 'web',
        trackingProperties: {},
      },
      frameId: '0',
    };
    return expected;
  }
});
