import * as React from 'react';
import './component.sass';
import { ExtensionPlatform, ExtensionViewType, ExtensionAnchor, ExtensionObject, ExtensionInstallationAbilities, Configuration, ExtensionMode } from 'extension-coordinator';
import { fetchHostingStatus, fetchGlobalConfigurationSegment, fetchChannelConfigurationSegments } from '../../util/api';
import { RigContext, RigExtensionView, getMode, ExtensionFrameOptions } from '../../core/models/rig';

const EXTENSION_FRAME_INIT_ACTION = 'extension-frame-init';
const EXTENSION_FRAME_VISIBILITY_ACTION = 'extension-frame-visibility';
const IFRAME_CLASS = 'extension-frame';
const extensionViewTypes = [
  ExtensionViewType.Component,
  ExtensionViewType.Config,
  ExtensionViewType.LiveConfig,
  ExtensionViewType.Mobile,
  ExtensionViewType.VideoOverlay,
  ExtensionViewType.Panel,
];

export interface Props {
  anchor: ExtensionAnchor;
  channelId: string;
  extension: ExtensionObject;
  frameId: string;
  hidden?: boolean;
  installationAbilities: ExtensionInstallationAbilities;
  isPopout: boolean;
  mode: ExtensionMode;
  platform: ExtensionPlatform;
  bindIframeToParent: (iframe: HTMLIFrameElement) => void;
}

export function createExtensionFrameProps(extension: ExtensionObject, view: RigExtensionView, bindIframeToParent: (iframe: HTMLIFrameElement) => void) {
  const { channelId, configurationAnchor, features, id, isPopout, label, type, viewerType } = view;
  const anchor = type === ExtensionMode.Config || type === ExtensionMode.Dashboard ?
    configurationAnchor : type as ExtensionAnchor;
  const isComponentOverlayMobile = [ExtensionAnchor.Component, ExtensionAnchor.Overlay, ExtensionPlatform.Mobile].some((t) => type === t);
  const mode = getMode(type);
  const extensionFrameProps = {
    anchor,
    channelId,
    extension,
    frameId: label || `${mode === ExtensionMode.Viewer ? `${viewerType} ${type}` : `${mode}`} ${id}`,
    installationAbilities: features,
    isPopout: !isComponentOverlayMobile && isPopout,
    mode: isComponentOverlayMobile ? ExtensionMode.Viewer : mode,
    platform: type === ExtensionPlatform.Mobile ? ExtensionPlatform.Mobile : ExtensionPlatform.Web,
    bindIframeToParent,
  };
  return extensionFrameProps;
}

export class ExtensionFrame extends React.Component<Props> {
  static contextType = RigContext;
  context!: React.ContextType<typeof RigContext>;
  private iframe?: HTMLIFrameElement;

  public componentWillUnmount() {
    if (this.iframe) {
      this.iframe.onload = null;
    }
  }

  public componentDidUpdate(prevProps: Props) {
    if (this.props.hidden !== prevProps.hidden && this.iframe) {
      this.iframe.contentWindow!.postMessage({
        action: EXTENSION_FRAME_VISIBILITY_ACTION,
        isVisible: Boolean(!this.props.hidden),
      }, '*');
    }
  }

  public render() {
    const { frameId } = this.props;
    /* eslint no-restricted-globals: "off" */
    const src = new URL('extension-frame.html', location.href);
    /* eslint no-restricted-globals: "error" */
    return (
      <iframe
        className={`rig-frame ${IFRAME_CLASS}`}
        frameBorder={0}
        src={src.href}
        title={frameId}
        ref={this.bindIframeRef}
      />
    );
  }

  private bindIframeRef = (iframe: HTMLIFrameElement) => {
    this.iframe = iframe;
    this.props.bindIframeToParent(iframe);
    if (this.iframe) {
      this.iframe.onload = this.extensionFrameInit;
    }
  }

  private async getConfiguration(): Promise<Configuration | undefined> {
    const { isConfigurationHosted, secret, userId } = this.context;
    if (isConfigurationHosted) {
      const { channelId, extension: { clientId } } = this.props;
      const globalSegment = await fetchGlobalConfigurationSegment(clientId, userId, secret);
      const segments = await fetchChannelConfigurationSegments(clientId, userId, channelId, secret);
      return {
        broadcaster: segments.broadcaster,
        developer: segments.developer,
        global: globalSegment,
      };
    }
  }

  private extensionFrameInit = async () => {
    const extension = JSON.parse(JSON.stringify(this.props.extension));
    const hostingStatus = await fetchHostingStatus();
    const configuration = await this.getConfiguration();

    // Avoid a race condition in which the iframe is now null.
    if (!this.iframe) {
      return;
    }

    // If there is a port, the Developer Rig is hosting the front end.  Set the
    // viewer URL of all extension views to use unencrypted localhost and that port.
    if (hostingStatus.port) {
      extensionViewTypes.forEach((viewType) => {
        const view = extension.views[viewType];
        if (view) {
          view.viewerUrl = `http://localhost:${hostingStatus.port}/${view.viewerUrl.split('/').slice(3).join('/')}`;
        }
      });
    }

    const { anchor, channelId, frameId, installationAbilities, isPopout, mode, platform } = this.props;
    const extensionFrameOptions: ExtensionFrameOptions = {
      anchor,
      channelId: Number(channelId),
      configuration,
      extension,
      iframeClassName: IFRAME_CLASS,
      installationAbilities,
      isPopout,
      loginId: null,
      mode,
      platform,
      trackingProperties: {},
    };
    const data = {
      action: EXTENSION_FRAME_INIT_ACTION,
      channelId,
      frameId,
      parameters: extensionFrameOptions,
    };

    this.iframe.contentWindow!.postMessage(data, '*');
  }
}
