import { setupShallowTest } from '../tests/enzyme-util/shallow';
import { OpenProjectDialog } from './component';
import { ExtensionManifest } from '../core/models/manifest';
import { RigProject } from '../core/models/rig';
import { timeout, ChangeEventFn } from '../tests/utils';
import { ShallowWrapper } from 'enzyme';

Math.random = () => .25;

function mockApiFunctions() {
  const original = require.requireActual('../util/api');
  return {
    ...original,
    fetchExtensionManifest: jest.fn().mockImplementation((id, _version) => id === 'error' ? Promise.reject(new Error('error')) : Promise.resolve({ id })),
  }
}
jest.mock('../util/api', () => mockApiFunctions());
const api = require.requireMock('../util/api');

describe('<OpenProjectDialog />', () => {
  const setupShallow = setupShallowTest(OpenProjectDialog, () => ({
    filePath: 'filePath',
    currentProject: {
      manifest: {
        id: 'id',
        version: 'version',
      } as ExtensionManifest,
      name: 'name',
    } as RigProject,
    userId: 'userId',
    closeHandler: jest.fn(),
    openHandler: jest.fn()
  }));

  it('renders correctly', () => {
    const { wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('InputProperty').forEach((node: ShallowWrapper) => node.dive());
    wrapper.find('Dialog').dive().find('Property').forEach((node: ShallowWrapper) => node.dive());
    expect(wrapper.debug()).toMatchSnapshot();
  });

  it('onChange sets state', async () => {
    const { wrapper } = setupShallow();
    const name = 'secret';
    const expectedValue = 'newSecret';
    wrapper.find(`Input[name="${name}"]`).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value: expectedValue } });
    expect(wrapper.state('secret')).toBe(expectedValue);
  });

  it('invokes closeHandler when top exit button is clicked', () => {
    const { props, wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalHeader').dive().find('CoreDismissible').simulate('click');
    expect(props.closeHandler).toHaveBeenCalled();
  });

  it('invokes closeHandler when cancel button is clicked', () => {
    const { props, wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive().find('Button').first().simulate('click');
    expect(props.closeHandler).toHaveBeenCalled();
  });

  it('invokes openHandler when open button is clicked', async () => {
    const value = 'value6789012345678901234567890123456789012345';
    const { props, wrapper } = setupShallow();
    await timeout();
    wrapper.find('Input[name="secret"]').prop<ChangeEventFn>('onChange')({ currentTarget: { name: 'secret', value } });
    const dive = wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive();
    dive.find('Button').last().simulate('click');
    await timeout();
    expect(api.fetchExtensionManifest).toHaveBeenCalledTimes(1);
    expect(props.openHandler).toHaveBeenCalledWith(props.currentProject, props.filePath, value);
  });

  it('does not invoke openHandler when open button is clicked', async () => {
    const { props, wrapper } = setupShallow();
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive().find('Button').last().simulate('click');
    await timeout();
    expect(props.openHandler).not.toHaveBeenCalled();
  });

  it('fails to open upon fetchExtensionManifest failure', async () => {
    const value = 'value6789012345678901234567890123456789012345';
    const { props, wrapper } = setupShallow();
    await timeout();
    const dive = wrapper.find('Dialog').dive();
    dive.find('Input[name="secret"]').prop<ChangeEventFn>('onChange')({ currentTarget: { name: 'secret', value } });
    let { error } = console;
    console.error = jest.fn();
    props.currentProject.manifest.id = 'error';
    wrapper.find('Dialog').dive().find('Modal').dive().find('ModalFooter').dive().find('Button').last().simulate('click');
    await timeout();
    expect(wrapper.state('saveResult')).toBe(2);
    [error, console.error] = [console.error, error];
    expect(error).toHaveBeenCalledWith('error');
  });
});
