import * as React from 'react';
import './component.sass';
import { RigProject } from '../core/models/rig';
import { fetchExtensionManifest } from '../util/api';
import { Dialog } from '../dialog';
import { CoreLink, CoreText, FormGroup, Input, InputType, SVG, SVGAsset, SVGType, TextType } from 'twitch-core-ui';

interface Props {
  filePath: string;
  currentProject: RigProject;
  userId: string;
  closeHandler: () => void;
  openHandler: (currentProject: RigProject, filePath: string, secret: string) => void;
}

enum Result {
  None = 0,
  Fetching = 1,
  Error = 2,
  Success = 3,
}

interface State {
  saveResult: Result;
  secret: string;
  [key: string]: Result | string;
}

const minimumSecretLength = 40;

export class OpenProjectDialog extends React.Component<Props, State>{
  public state: State = {
    saveResult: Result.None,
    secret: '',
  };

  private canOpen = () => {
    // There must be a secret to verify.
    if (!this.state.secret) {
      return false;
    }

    // The secret must be an extension secret.
    if (this.state.secret.length < minimumSecretLength) {
      return false;
    }

    return true;
  }

  private onChange = (input: React.FormEvent<HTMLInputElement>) => {
    const { name, value } = input.currentTarget;
    this.setState({ [name]: value.trim(), saveResult: Result.None });
  }

  private open = async () => {
    if (this.canOpen()) {
      try {
        this.setState({ saveResult: Result.Fetching });
        const { currentProject, filePath } = this.props;
        const { id: clientId, version } = currentProject.manifest;
        const { secret } = this.state;
        await fetchExtensionManifest(clientId, version);
        this.props.openHandler(currentProject, filePath, secret);
      } catch (ex) {
        console.error(ex.message);
        this.setState({ saveResult: Result.Error });
      }
    }
  }

  public render() {
    const { closeHandler, currentProject } = this.props;
    const { saveResult, secret } = this.state;
    return <>
      <Dialog className="open-project-dialog" canConfirm={this.canOpen()} confirmText="Open" title="Open Project"
        closeHandler={closeHandler} confirmHandler={this.open}>
        <FormGroup hint="This is the name of your project." label="Project Name">
          <CoreText>{currentProject.name}</CoreText>
        </FormGroup>
        <FormGroup hint="This is the client ID of your extension." label="Client ID">
          <CoreText>{currentProject.manifest.id}</CoreText>
        </FormGroup>
        <FormGroup label="Extension Secret" hint={`Extension secrets are at least ${minimumSecretLength} characters long.  API client secrets are shorter and not for this purpose.`}>
          <CoreLink ariaLabel="OpenProjectDialog:Retrieve your Extension Secret" linkTo={`https://dev.twitch.tv/console/extensions/${currentProject.manifest.id}/authentication-settings`} targetBlank>Retrieve your Extension Secret</CoreLink>
          <Input error={!secret.trim()} type={InputType.Text} name="secret" placeholder="Enter your extension's secret" value={secret} onChange={this.onChange} />
        </FormGroup>
        <div className="open-project-dialog__save-result">
          {saveResult === Result.Fetching ? (
            <CoreText type={TextType.Span}>Verifying...</CoreText>
          ) : saveResult === Result.Error ? <>
            <SVG asset={SVGAsset.NotificationError} type={SVGType.Alert} />
            <CoreText type={TextType.Span}>Invalid secret</CoreText>
          </> : <div />}
        </div>
      </Dialog>
    </>;
  }
}
