import * as React from 'react';
import { shallow } from 'enzyme';
import { ProcessView } from './component';
import { timeout } from '../tests/utils';
import electron from '../electron';

const setupShallow = () => shallow(<ProcessView />);

describe('<ProcessView />', () => {
  it('renders', async () => {
    const wrapper = setupShallow();
    const instance = wrapper.instance() as any;
    const backendScrollHeight = 1;
    const frontendScrollHeight = 2;
    instance.backendView.current = { scrollHeight: backendScrollHeight };
    instance.frontendView.current = { scrollHeight: frontendScrollHeight };
    await timeout();
    expect(wrapper.debug()).toMatchSnapshot();
    expect(instance.backendView.current.scrollTop).toBe(backendScrollHeight);
    expect(instance.frontendView.current.scrollTop).toBe(frontendScrollHeight);
    expect(electron.ipcRenderer.on).toHaveBeenCalledTimes(2);
    wrapper.unmount();
    expect(electron.ipcRenderer.removeListener).toHaveBeenCalledTimes(2);
  });

  it('adds to stderr in back end', async () => {
    await stderr('back-end');
  });

  it('adds to stderr in front end', async () => {
    await stderr('front-end');
  });

  it('adds to stdout in back end', async () => {
    await stdout('back-end');
  });

  it('adds to stdout in front end', async () => {
    await stdout('front-end');
  });

  async function stderr(source: string) {
    const wrapper = setupShallow();
    const instance = wrapper.instance() as any;
    await timeout();
    let { error } = console;
    console.error = jest.fn();
    const message = '\x1b[37mtest\x1b[0m error';
    instance.onStderr(undefined, source, message);
    [error, console.error] = [console.error, error];
    expect(error).toHaveBeenCalledWith(source, message);
  }

  async function stdout(source: string) {
    const wrapper = setupShallow();
    const instance = wrapper.instance() as any;
    await timeout();
    let { log } = console;
    console.log = jest.fn();
    const message = '\x1b[37mtest\x1b[0m message';
    instance.onStdout(undefined, source, message);
    [log, console.log] = [console.log, log];
    expect(log).toHaveBeenCalledWith(source, message);
  }
});
