import * as React from 'react';
import './component.sass';
import electron from '../electron';
import classNames from 'classnames';
import { Background, CoreText, Display, JustifyContent, StyledLayout } from 'twitch-core-ui';

interface Entry {
  isError: boolean;
  text: string;
}

interface State {
  backendEntries: Entry[];
  frontendEntries: Entry[];
}

// https://en.wikipedia.org/wiki/ANSI_escape_code
const ColorSpecifications = (() => {
  const makeColor = (i: number, ch: string) => '#' + (8 + i).toString(2).substr(1).replace(/1/g, ch).split('').reverse().join('');
  return [...Array(7).keys()].map((i) => i + 1).reduce((colorSpecifications: { [key: string]: string }, i: number) => {
    colorSpecifications[`3${i}`] = makeColor(i, 'a');
    colorSpecifications[`9${i}`] = makeColor(i, 'f');
    return colorSpecifications;
  }, { '90': '#777' });
})();

export class ProcessView extends React.Component<{}, State> {
  private backendView: React.RefObject<HTMLDivElement>;
  private frontendView: React.RefObject<HTMLDivElement>;

  public state: State = {
    backendEntries: [],
    frontendEntries: [],
  };

  private clearBackendLog = (_event: React.MouseEvent) => {
    this.setState({ backendEntries: [] });
  }

  private clearFrontendLog = (_event: React.MouseEvent) => {
    this.setState({ frontendEntries: [] });
  }

  constructor(props: {}) {
    super(props);
    this.backendView = React.createRef();
    this.frontendView = React.createRef();
  }

  public componentDidMount() {
    electron.ipcRenderer.on('stderr', this.onStderr);
    electron.ipcRenderer.on('stdout', this.onStdout);
  }

  public componentWillUnmount() {
    electron.ipcRenderer.removeListener('stderr', this.onStderr);
    electron.ipcRenderer.removeListener('stdout', this.onStdout);
  }

  private onStderr = (_event: any, source: string, message: string) => {
    this.addEntry(source, message, true);
    console.error(source, message);
  }

  private onStdout = (_event: any, source: string, message: string) => {
    this.addEntry(source, message, false);
    console.log(source, message);
  }

  private addEntry(source: string, text: string, isError: boolean) {
    this.setState((previousState) => {
      if (source === 'back-end') {
        return { backendEntries: [...previousState.backendEntries, { isError, text }], frontendEntries: previousState.frontendEntries };
      } else {
        return { backendEntries: previousState.backendEntries, frontendEntries: [...previousState.frontendEntries, { isError, text }] };
      }
    });
  }

  private formatEntries(entries: Entry[]) {
    return entries.map((entry, index) => {
      /*eslint no-control-regex: "off"*/
      const text = entry.text.replace(/\r/g, String.fromCharCode(1)).replace(/\n/g, String.fromCharCode(2));
      const parts = [text.replace(/\x1b.*/, '').replace(/\x01/g, '\r').replace(/\x02/g, '\n')];
      const colorSpecifications: (string | undefined)[] = [undefined];
      const replace = (_match: string, colorSpecification: string, rest: string, _offset: number, _whole: string): string => {
        parts.push(rest.replace(/\x01/g, '\r').replace(/\x02/g, '\n'));
        colorSpecifications.push(ColorSpecifications[colorSpecification]);
        return '';
      };
      text.replace(/[^\x1b]*/, '').replace(/\x1b\[([^m]*)m([^\x1b]*)/g, replace);
      const spans = parts.map((part, index) => part ? (
        <span key={index} style={colorSpecifications[index] ? { color: colorSpecifications[index] } : undefined}>{part}</span>
      ) : undefined);
      const className = classNames('process-view__output', {
        'process-view__output--error': entry.isError,
      });
      return <pre key={index} className={className}>{spans}</pre>;
    });
  }

  private getBackendView() {
    setTimeout(() => {
      const { current } = this.backendView;
      if (current) {
        current.scrollTop = current.scrollHeight;
      }
    });
    const { backendEntries, frontendEntries } = this.state;
    const sectionClassName = classNames('process-view__section', {
      'process-view__section--half': frontendEntries.length,
    });
    const entriesClassName = classNames('process-view__entries', {
      'process-view__entries--half': frontendEntries.length,
    });
    const entries = this.formatEntries(backendEntries);
    return entries.length ? (
      <div className={sectionClassName}>
        <StyledLayout background={Background.Alt2} display={Display.Flex} justifyContent={JustifyContent.Between} padding={{ x: 0.5 }}>
          <CoreText>Back End</CoreText>
          <button onClick={this.clearBackendLog}>Clear</button>
        </StyledLayout>
        <div className={entriesClassName} ref={this.backendView}>
          {entries}
        </div>
      </div>
    ) : null;
  }

  private getFrontendView() {
    setTimeout(() => {
      const { current } = this.frontendView;
      if (current) {
        current.scrollTop = current.scrollHeight;
      }
    });
    const { backendEntries, frontendEntries } = this.state;
    const sectionClassName = classNames('process-view__section', {
      'process-view__section--half': backendEntries.length,
    });
    const entriesClassName = classNames('process-view__entries', {
      'process-view__entries--half': backendEntries.length,
    });
    const entries = this.formatEntries(frontendEntries);
    return entries.length ? (
      <div className={sectionClassName}>
        <StyledLayout background={Background.Alt2} display={Display.Flex} justifyContent={JustifyContent.Between} padding={{ x: 0.5 }}>
          <CoreText>Front End</CoreText>
          <button onClick={this.clearFrontendLog}>Clear</button>
        </StyledLayout>
        <div className={entriesClassName} ref={this.frontendView}>
          {this.formatEntries(frontendEntries)}
        </div>
      </div>
    ) : null;
  }

  public render() {
    return (
      <div className="process-view">
        {this.getBackendView()}
        {this.getFrontendView()}
      </div>
    );
  }
}
