import { setupShallowTest } from '../tests/enzyme-util/shallow';
import { ProductsView } from './component';
import { ProductRow } from './product-row';
import * as TestData from '../tests/constants/products';
import { timeout, ChangeEventFn } from '../tests/utils';
import * as ProductErrors from '../constants/product-errors';

function mockApiFunctions() {
  const original = require.requireActual('../util/api');
  return {
    ...original,
    fetchProducts: jest.fn().mockImplementation((clientId, _token) => Promise.resolve([{ ...TestData.TestProduct1 }, { ...(TestData as any)[clientId] }])),
    saveProduct: jest.fn().mockImplementation((_clientId, token, _product) => token === 'error' ? Promise.reject(new Error('error')) : Promise.resolve()),
  }
}
jest.mock('../util/api', () => mockApiFunctions());
const api = require.requireMock('../util/api');

describe('<ProductsView />', () => {
  const setupRenderer = function(name = 'TestProduct2', token = name) {
    const fn = setupShallowTest(ProductsView, () => ({ clientId: name, isBitsEnabled: true, token, version: '0.0.1' }));
    return fn();
  }

  it('renders correctly', async () => {
    const { wrapper } = setupRenderer();
    await timeout();
    expect(wrapper.find('ProductRow').first().dive().find('ProductSelect').first().dive().find('Select')).toHaveLength(1);
    expect(wrapper.debug()).toMatchSnapshot();
    expect(api.fetchProducts).toHaveBeenCalledTimes(1);
    expect(wrapper.state('products')).toHaveLength(2);
  });

  it('displays failure upon product fetch failure', async () => {
    let fetchProducts = api.fetchProducts;
    api.fetchProducts = jest.fn(() => Promise.reject(new Error('error')));
    const { wrapper } = setupRenderer();
    await timeout();
    [fetchProducts, api.fetchProducts] = [api.fetchProducts, fetchProducts];
    expect(fetchProducts).toHaveBeenCalledTimes(1);
    expect(wrapper.state('message')).toBe('Error fetching products: error');
  });

  it('displays deprecated products', async () => {
    const { wrapper } = setupRenderer();
    await timeout();
    wrapper.find('.products-view__tab').at(1).simulate('click');
    expect(wrapper.state('showingDeprecated')).toBe(true);
    wrapper.find('.products-view__tab').first().simulate('click');
    expect(wrapper.state('showingDeprecated')).toBe(false);
  });

  it('adds a product when the Add Product button is clicked', async () => {
    const { wrapper } = setupRenderer();
    await timeout();
    wrapper.find('Button').first().simulate('click');
    expect(wrapper.state('products')).toHaveLength(3);
  });

  it('changes product values when editing products', async () => {
    const { wrapper } = setupRenderer();
    await timeout();
    const fieldName = 'sku';
    const value = 'newSku';
    const productRowWrapper = wrapper.find(ProductRow).first().dive();
    productRowWrapper.find(`Input[name="${fieldName}"]`).prop<ChangeEventFn>('onChange')({ currentTarget: { name: fieldName, value: value } });
    const products = wrapper.state('products') as Array<{ [key: string]: string }>;
    expect(products[0][fieldName]).toBe(value);
  });

  it('deprecates a product when the Deprecate button is clicked', async () => {
    const { wrapper } = setupRenderer();
    await timeout();
    const productRowWrapper = wrapper.find(ProductRow).first().dive();
    productRowWrapper.find('Button').simulate('click');
    const products = wrapper.state('products') as Array<{ deprecated: boolean }>;
    expect(products[0].deprecated).toBeTruthy();
  });

  it('saves dirty product when the Save All button is clicked', async () => {
    api.saveProduct.mockClear();
    const { wrapper } = setupRenderer('UnsavedProduct');
    await timeout();
    wrapper.find('Button').last().simulate('click');
    expect(api.saveProduct).toHaveBeenCalled();
  });

  it('fails save', async () => {
    api.saveProduct.mockClear();
    const { wrapper } = setupRenderer('UnsavedProduct', 'error');
    await timeout();
    wrapper.find('Button').last().simulate('click');
    await timeout();
    expect(api.saveProduct).toHaveBeenCalled();
    expect(wrapper.state('message')).toBe('Error saving products: error');
  });

  it('disables save button when display name is empty', async () => {
    await testSave('displayName', '', ProductErrors.NAME_EMPTY);
  });

  it('disables save button when display name is too long', async () => {
    await testSave('displayName', Array(333).fill('-').join(''), ProductErrors.NAME_CHAR_LIMIT);
  });

  it('disables save button when SKU is empty', async () => {
    await testSave('sku', '', ProductErrors.SKU_EMPTY);
  });

  it('disables save button when SKU contains whitespace', async () => {
    await testSave('sku', '- -', ProductErrors.SKU_WHITESPACE);
  });

  it('disables save button when SKU is too long', async () => {
    await testSave('sku', Array(333).fill('-').join(''), ProductErrors.SKU_CHAR_LIMIT);
  });

  it('disables save button when amount is empty', async () => {
    await testSave('amount', '', ProductErrors.AMOUNT_EMPTY);
  });

  it('disables save button when amount is too large', async () => {
    await testSave('amount', '11111', ProductErrors.AMOUNT_OUT_OF_RANGE);
  });

  it('disables save button when a product is invalid', async () => {
    const { wrapper } = setupRenderer('ChangedInvalidProduct');
    await timeout();
    expect(wrapper.find('Button').last().prop('disabled')).toBeTruthy();
  });

  it('disables save button when duplicate SKUs are found', async () => {
    const { wrapper } = setupRenderer('TestProduct1');
    await timeout();
    expect(wrapper.find('Button').last().prop('disabled')).toBeTruthy();
  });

  async function testSave(fieldName: string, fieldValue: string, productError: string) {
    const { wrapper } = setupRenderer();
    await timeout();
    const productRowWrapper = wrapper.find(ProductRow).first().dive();
    productRowWrapper.find(`Input[name="${fieldName}"]`).prop<ChangeEventFn>('onChange')({ currentTarget: { name: fieldName, value: fieldValue } });
    const products = wrapper.state('products') as Array<{ validationErrors: { [key: string]: string } }>;
    expect(products[0].validationErrors[fieldName]).toBe(productError);
    expect(wrapper.find('Button').last().prop('disabled')).toBeTruthy();
  }
});

describe('<ProductRow />', () => {
  const setupShallow = setupShallowTest((ProductRow as any), () => ({
    product: TestData.TestProduct1,
  }));

  it('renders correctly', () => {
    const { wrapper } = setupShallow();
    expect(wrapper.debug()).toMatchSnapshot();
  });

  it('shows validation errors', () => {
    const { wrapper } = setupShallow({
      product: {
        ...TestData.TestProduct1,
        validationErrors: {
          sku: 'SKU is invalid'
        }
      }
    });
    expect(wrapper.find('Input[name="sku"]').prop('error')).toBeTruthy();
    expect(wrapper.find('CoreText').filterWhere(n => n.html().includes('SKU is invalid'))).toHaveLength(1);
  });

  it('shows row as dirty', () => {
    const { wrapper } = setupShallow({
      product: TestData.UnsavedProduct,
    });
    expect(wrapper.find('StyledLayout[visibility="hidden"]')).toHaveLength(0);
  });
});
