import * as React from 'react';
import './component.sass';
import * as ProductErrors from '../constants/product-errors';
import { ProductRow } from './product-row';
import { fetchProducts, saveProduct } from '../util/api';
import { Product, ValidationErrors } from '../core/models/product';
import classNames from 'classnames';
import { CoreText, Button, ButtonType, SVGAsset, Layout, Display, JustifyContent, Position, Balloon, BalloonSize, BalloonDirection, Background, Color, StyledLayout, MarginValues, AlignItems, FontSize, CoreLink } from 'twitch-core-ui';

const maximumProductCount = 250;

export interface Props {
  clientId: string;
  isBitsEnabled: boolean;
  token: string;
  version: string;
}

export interface State {
  products: Product[];
  message: string;
  showingDeprecated: boolean;
}

export class ProductsView extends React.Component<Props, State>{
  state: State = {
    products: [],
    message: '',
    showingDeprecated: false,
  };

  public async componentDidMount() {
    try {
      const { clientId, token } = this.props;
      const products = await fetchProducts(clientId, token);
      this.setState({ products });
    } catch (ex) {
      this.setState({ message: `Error fetching products: ${ex.message}` });
    }
  }

  public onChange(index: number, event: React.FormEvent<HTMLInputElement>) {
    const { name, value } = event.currentTarget;
    this.changeProductValue(index, name, value);
  }

  public deprecateProduct(index: number) {
    const deprecated = this.state.products[index].deprecated;
    this.changeProductValue(index, 'deprecated', !deprecated);
  }

  public addProduct = () => {
    if (this.state.products.length < maximumProductCount) {
      const product = {
        displayName: 'New Product',
        sku: 'newSKU',
        amount: 1,
        inDevelopment: true,
        broadcast: true,
        deprecated: false,
        dirty: true,
        savedInCatalog: false,
        error: '',
        validationErrors: {},
      };
      this.state.products.push(product);
      this.setState({ products: this.state.products });
    }
  }

  public saveProducts = async () => {
    this.setState({ message: 'saving...' });
    try {
      const dirtyProducts = this.state.products.filter((product) => product.dirty);
      const { clientId, token } = this.props;
      await Promise.all(dirtyProducts.map(async (product) => {
        await saveProduct(clientId, token, product);
        product.savedInCatalog = true;
        product.dirty = false;
      }));
      this.setState({ message: '' });
    } catch (ex) {
      this.setState({ message: `Error saving products: ${ex.message}` });
    }
  }

  public render() {
    if (!this.props.isBitsEnabled) {
      return <Layout alignItems={AlignItems.Center} display={Display.Flex} fullHeight justifyContent={JustifyContent.Center}>
        <CoreText color={Color.Base} fontSize={FontSize.Size4}>Bits are not enabled.  Visit
          the <CoreLink ariaLabel="ProductsView:Monetization section" linkTo={`https://dev.twitch.tv/console/extensions/${this.props.clientId}/${this.props.version}/monetization`} targetBlank>Monetization section</CoreLink> of
          this version of your extension to learn more.</CoreText>
      </Layout>;
    }

    const skus = this.state.products.map(p => p.sku);
    const isAddDisabled = this.state.products.length >= maximumProductCount;
    let isSaveDisabled = false;
    const liveProducts: JSX.Element[] = [];
    const deprecatedProducts: JSX.Element[] = [];

    this.state.products.forEach((p, i) => {
      const matchingSkus = skus.filter(sku => sku === p.sku);
      p.validationErrors = p.validationErrors || {};
      if (matchingSkus.length > 1) {
        p.validationErrors = {
          ...p.validationErrors,
          sku: ProductErrors.SKU_UNIQUE,
        };
      } else if (p.validationErrors.sku === ProductErrors.SKU_UNIQUE) {
        delete p.validationErrors.sku;
      }

      if (Object.keys(p.validationErrors).length > 0) {
        isSaveDisabled = true;
      }

      const productRowElement = (
        <ProductRow key={i} product={p}
          onChange={this.onChange.bind(this, i)}
          onDeprecate={this.deprecateProduct.bind(this, i)}
        />
      );

      if (p.deprecated) {
        deprecatedProducts.push(productRowElement);
      } else {
        liveProducts.push(productRowElement);
      }
    });
    const liveClassName = classNames('products-view__tab', {
      'products-view__tab--selected': !this.state.showingDeprecated,
    });
    const deprecatedClassName = classNames('products-view__tab', {
      'products-view__tab--selected': this.state.showingDeprecated,
    });
    const ProductHeader: React.FunctionComponent<{ headerText: string; ballonText: string; }> = (props) => (
      <Layout className="products-view__header" margin={{ top: 1 }} position={Position.Relative}>
        <Layout margin={props.headerText.length < 10 ? { right: Math.min(5, 10 - props.headerText.length) } as MarginValues : undefined}>
          <CoreText bold color={Color.Base} noWrap>{props.headerText}</CoreText>
        </Layout>
        <div className="products-view__header-balloon">
          <Balloon direction={BalloonDirection.BottomLeft} show size={BalloonSize.Small} tailBackground={Background.AccentAlt2}>
            <StyledLayout background={Background.AccentAlt2} padding={1}>
              <CoreText color={Color.Overlay}>{props.ballonText}</CoreText>
            </StyledLayout>
          </Balloon>
        </div>
      </Layout>
    );

    return (
      <div className="products-view">
        <Layout display={Display.Flex}>
          <div className={liveClassName} onClick={() => this.setState({ showingDeprecated: false })}>Live Products</div>
          <div className={deprecatedClassName} onClick={() => this.setState({ showingDeprecated: true })}>Deprecated Products</div>
          <Layout className="products-view__tab" flexGrow={1} />
        </Layout>
        <div className="products-view__content">
          <ProductHeader headerText="Product Name" ballonText="Name to display for this product in your extension." />
          <ProductHeader headerText="SKU" ballonText="Unique SKU to identify this product.  Cannot be changed after saving.  May not contain whitespace." />
          <ProductHeader headerText="Amount (in Bits)" ballonText="Amount of Bits to offer this product for.  Must be between 1 and 10,000." />
          <ProductHeader headerText="In Development" ballonText="Setting this to Yes will cause this product to not be displayed in a released extension." />
          <ProductHeader headerText="Broadcast" ballonText="Setting this to Yes will notify all instances of your extension on a channel when a transaction for this product has completed." />
          <div />
          <div />
          {this.state.showingDeprecated ? deprecatedProducts : liveProducts}
        </div>
        <Layout className="products-view__buttons">
          <CoreText color={Color.Base}>{this.state.message}</CoreText>
          {this.state.showingDeprecated ? <div /> : (
            <Button ariaLabel="ProductsView:Add New Product" disabled={isAddDisabled} icon={SVGAsset.AddReaction} type={ButtonType.Hollow} onClick={this.addProduct}>Add New Product</Button>
          )}
          <Button ariaLabel="ProductsView:Save All" disabled={isSaveDisabled} onClick={this.saveProducts}>Save All</Button>
        </Layout>
      </div>
    );
  }

  private changeProductValue(index: number, fieldName: string, value: boolean | string) {
    const product = this.state.products[index];
    (product as any)[fieldName] = value;
    product.dirty = true;
    product.validationErrors = this.validateProduct(product);
    this.setState({ products: this.state.products });
  }

  private validateProduct(product: Product): ValidationErrors {
    const validationErrors: ValidationErrors = {};
    if (!product.displayName) {
      validationErrors.displayName = ProductErrors.NAME_EMPTY;
    } else if (product.displayName.length > 255) {
      validationErrors.displayName = ProductErrors.NAME_CHAR_LIMIT;
    }
    if (!product.sku) {
      validationErrors.sku = ProductErrors.SKU_EMPTY;
    } else if (product.sku.search(/^\S*$/)) {
      validationErrors.sku = ProductErrors.SKU_WHITESPACE;
    } else if (product.sku.length > 255) {
      validationErrors.sku = ProductErrors.SKU_CHAR_LIMIT;
    }
    if (!product.amount) {
      validationErrors.amount = ProductErrors.AMOUNT_EMPTY;
    } else if (product.amount < 1 || product.amount > 10000) {
      validationErrors.amount = ProductErrors.AMOUNT_OUT_OF_RANGE;
    }
    return validationErrors;
  }
}
