import * as React from 'react';
import { Product } from '../core/models/product';
import { AlignItems, Button, ButtonType, Color, CoreText, Display, Input, InputType, JustifyContent, Select, StyledLayout, Visibility } from 'twitch-core-ui';

enum SelectValue {
  No = 'No',
  Yes = 'Yes',
}

interface Props {
  product: Product;
  onChange: (event: any) => void;
  onDeprecate: () => void;
}

export class ProductRow extends React.Component<Props>{
  private onSelectChange = (event: React.FormEvent<HTMLSelectElement>) => {
    const { name, value } = event.currentTarget;
    this.props.onChange({ currentTarget: { name, value: value === SelectValue.Yes } });
  }

  render() {
    const ProductSelect = ({ name, value }: { name: string, value: boolean }) => (
      <Select name={name} value={value ? SelectValue.Yes : SelectValue.No} onChange={this.onSelectChange}>
        <option value={SelectValue.Yes}>Yes</option>
        <option value={SelectValue.No}>No</option>
      </Select>
    );
    const { product, onChange, onDeprecate } = this.props;
    const validationErrors = product.validationErrors || {};
    const isInvalid = Object.keys(validationErrors).length > 0;
    const visibility = product.dirty ? Visibility.Visible : Visibility.Hidden;

    return <>
      <Input type={InputType.Text}
        error={!!validationErrors.displayName}
        name="displayName"
        required
        maxLength={255}
        value={product.displayName}
        onChange={onChange}
      />
      <Input type={InputType.Text}
        error={!!validationErrors.sku}
        name="sku"
        required
        maxLength={255}
        disabled={product.savedInCatalog}
        value={product.sku}
        onChange={onChange}
      />
      <Input type={InputType.Number}
        error={!!validationErrors.amount}
        name="amount"
        required
        min={1} max={10000}
        value={product.amount.toString()}
        onChange={onChange}
      />
      <ProductSelect name="inDevelopment" value={product.inDevelopment} />
      <ProductSelect name="broadcast" value={product.broadcast} />
      <Button ariaLabel={`ProductRow:${product.deprecated ? 'Restore' : 'Deprecate'}`} fullWidth type={ButtonType.Hollow} onClick={onDeprecate}>
        {product.deprecated ? 'Restore' : 'Deprecate'}
      </Button>
      <StyledLayout alignItems={AlignItems.Center} color={Color.Link} display={Display.Flex} fullHeight justifyContent={JustifyContent.Center} visibility={visibility}>
        <svg xmlns="http://www.w3.org/2000/svg" fill="currentColor" height="6" width="6">
          <circle cx="3" cy="3" r="3" />
        </svg>
      </StyledLayout>
      {isInvalid && ['displayName', 'sku', 'amount', '', '', '', ''].map((name, index) => (
        <div className="products-view__invalid-hint" key={`hint${index}`}>
          <CoreText color={Color.Error}>{(validationErrors as { [key: string]: string })[name]}</CoreText>
        </div>
      ))}
    </>;
  }
}
