import { setupShallowTest } from '../tests/enzyme-util/shallow';
import { ProjectView } from './component';
import { createExtensionManifestForTest } from '../tests/constants/extension';
import { RigExtensionView } from '../core/models/rig';
import { timeout, ChangeEventFn } from '../tests/utils';
import { ShallowWrapper } from 'enzyme';

function mockApiFunctions() {
  const original = require.requireActual('../util/api');
  return {
    ...original,
    fetchExtensionManifest: jest.fn().mockImplementation((id, _version) => id === 'error' ? Promise.reject(new Error('error')) : Promise.resolve({ state: 'Testing' })),
    fetchExtensionSecret: jest.fn().mockImplementation((id) => id === 'error' ? Promise.reject(new Error('error')) : Promise.resolve('secret')),
    fetchHostingStatus: jest.fn().mockImplementation(() => Promise.resolve({})),
    hasVisualStudioCode: jest.fn().mockImplementation(() => Promise.resolve(true)),
    launchVisualStudioCode: jest.fn().mockImplementation(() => Promise.resolve({})),
    hostFrontend: jest.fn().mockImplementation(() => Promise.resolve({})),
    openUrl: jest.fn().mockImplementation(() => Promise.resolve({})),
    showOpenDialog: jest.fn().mockImplementation(() => Promise.resolve('test-test')),
    startBackend: jest.fn().mockImplementation(() => Promise.resolve({})),
    startFrontend: jest.fn().mockImplementation(() => Promise.resolve({})),
    stopHosting: jest.fn().mockImplementation(() => Promise.resolve({})),
  };
}
jest.mock('../util/api', () => mockApiFunctions());
const api = require.requireMock('../util/api');

describe('<ProjectView />', () => {
  const rigProject = {
    allowHttpBackend: false,
    backendCommand: 'test',
    backendFolderName: 'test',
    certificateExceptions: [],
    extensionViews: [] as RigExtensionView[],
    frontendCommand: 'test',
    frontendFolderName: 'test',
    manifest: createExtensionManifestForTest(),
    name: 'test',
    projectFolderPath: 'test',
    secret: 'test',
    usingRandomFrontendHostingPort: false,
    version: 1,
  };
  const setupShallow = setupShallowTest(ProjectView, () => ({
    rigProject: { ...rigProject },
    secret: 'test',
    userId: '999999999',
    closeProject: jest.fn(),
    onChange: jest.fn(),
    refreshViews: jest.fn(),
    showReleaseNotes: jest.fn(),
  }));

  it('renders correctly', () => {
    const { wrapper } = setupShallow();
    expect(wrapper.debug()).toMatchSnapshot();
    wrapper.find('InputProperty').forEach((node: ShallowWrapper) => node.dive());
    wrapper.find('Property').forEach((node: ShallowWrapper) => node.dive());
  });

  it('invokes openProjectFolder', () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    instance.openProjectFolder();
    expect(api.openUrl).toHaveBeenCalled();
  });

  it('launches VS Code', async () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    instance.launchVisualStudioCode();
    expect(api.launchVisualStudioCode).toHaveBeenCalled();
  });

  describe('onChange', () => {
    it('invokes', () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as ProjectView;
      const name = 'projectFolderPath';
      wrapper.find('InputProperty').at(2).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value: name } });
      expect(instance.props.onChange).toHaveBeenCalled();
    });

    it('invokes version', () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as ProjectView;
      const name = 'version';
      wrapper.find('InputProperty').at(1).prop<ChangeEventFn>('onChange')({ currentTarget: { name, value: name } });
      expect(instance.state.updateResult).toBe('');
      expect(instance.state.version).toBe('version');
    });
  });

  it('invokes fetchExtensionManifest through refreshManifest', async () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    await instance.refreshManifest();
    expect(api.fetchExtensionManifest).toHaveBeenCalled();
    expect(api.fetchExtensionSecret).toHaveBeenCalled();
  });

  it('invokes fetchExtensionManifest through updateByVersion', async () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    await instance.updateByVersion();
    expect(api.fetchExtensionManifest).toHaveBeenCalled();
  });

  it('updateByVersion fails', async () => {
    const { wrapper } = setupShallow();
    const instance = wrapper.instance() as any;
    instance.props.rigProject.manifest.id = 'error';
    await instance.updateByVersion();
    expect(wrapper.state('updateResult')).toBe('not found');
  });

  describe('front-end', () => {
    it('selects folder', async () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.selectFrontendFolder();
      await timeout();
      expect(instance.props.onChange).toBeCalled();
    });

    it('starts Developer Rig hosting', () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.props.rigProject.frontendCommand = '';
      instance.toggleFrontend();
      expect(api.hostFrontend).toHaveBeenCalled();
    });

    it('starts custom hosting', () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.toggleFrontend();
      expect(api.startFrontend).toHaveBeenCalled();
    });

    it('fails to start', async () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.props.rigProject.manifest.views = {};
      const errorMessage = 'errorMessage';
      api.startFrontend = jest.fn().mockImplementation(() => Promise.reject(new Error(errorMessage)));
      await instance.toggleFrontend();
      expect(instance.state.frontendResult).toBe(errorMessage);
    });

    it('stops', async () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.state.frontendResult = 'running';
      await instance.toggleFrontend();
      expect(api.stopHosting).toHaveBeenCalled();
    });
  });

  describe('back-end', () => {
    it('selects folder', async () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.selectBackendFolder();
      await timeout();
      expect(instance.props.onChange).toBeCalled();
    });

    it('starts', async () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      await instance.toggleBackend();
      expect(api.startBackend).toHaveBeenCalled();
    });

    it('fails to start', async () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.props.rigProject.manifest.views = {};
      const errorMessage = 'errorMessage';
      api.startBackend = jest.fn().mockImplementation(() => Promise.reject(new Error(errorMessage)));
      await instance.toggleBackend();
      expect(instance.state.backendResult).toBe(errorMessage);
    });

    it('stops', async () => {
      const { wrapper } = setupShallow();
      const instance = wrapper.instance() as any;
      instance.state.backendResult = 'running';
      await instance.toggleBackend();
      expect(api.stopHosting).toHaveBeenCalled();
    });
  });

  it('unmounts', async () => {
    const { wrapper } = setupShallow();
    await timeout();
    wrapper.unmount();
  });
});
