import * as React from 'react';
import './component.sass';
import { RigProject, HostingResult, toggleBackend, toggleFrontend } from '../core/models/rig';
import { fetchHostingStatus, showOpenDialog, setHttpOption, hasVisualStudioCode, launchVisualStudioCode, fetchExtensionManifest, fetchExtensionSecret, openUrl } from '../util/api';
import { NavItem } from '../rig-nav';
import { StyledLayout, Background, CoreText, TextType, FormGroup, Button, ButtonType, Layout, Display, BorderRadius, JustifyContent, Input, InputType, Position, SVGAsset, CheckBox, Color, Balloon, BalloonDirection, BalloonSize, CoreLink } from 'twitch-core-ui';
import { LocalStorage } from '../util/local-storage';

interface Props {
  rigProject: RigProject;
  userId: string;
  closeProject: () => void;
  onChange: (rigProject: RigProject) => void;
  refreshViews: () => void;
}

interface State {
  backendResult: string;
  frontendResult: string;
  hasVisualStudioCode?: boolean;
  isRefreshAware: boolean;
  updateResult: string;
  version: string;
}

interface PropertyProps {
  name: string;
  value: string;
}

type InputPropertyProps = PropertyProps & {
  action: {
    disabled?: boolean;
    result?: string;
    text: string;
    onClick: () => void;
  };
  disabled?: boolean;
  hint?: string;
  isDefault?: boolean;
  label: string;
  placeholder?: string;
  required?: boolean;
  onChange?: (event: React.FormEvent<HTMLInputElement>) => void;
};

export class ProjectView extends React.Component<Props, State>{
  public state: State = {
    backendResult: HostingResult.None,
    frontendResult: HostingResult.None,
    updateResult: '',
    isRefreshAware: true,
    version: this.props.rigProject.manifest.version,
  };

  async componentDidMount() {
    this.componentDidUpdate({} as Props);
    const value = await hasVisualStudioCode();
    const localStorage = new LocalStorage();
    this.setState({ hasVisualStudioCode: value, isRefreshAware: localStorage.isRefreshAware });
  }

  async componentDidUpdate(prevProps: Props) {
    if (prevProps.rigProject !== this.props.rigProject) {
      const status = await fetchHostingStatus();
      const { rigProject } = this.props;
      const getStatus = (value: string, isRunning: boolean): HostingResult => {
        return value ? isRunning ? HostingResult.Running : HostingResult.NotRunning : HostingResult.None;
      };
      this.setState({
        backendResult: getStatus(rigProject.backendCommand, status.isBackendRunning),
        frontendResult: status.isFrontendRunning ? HostingResult.Running : HostingResult.NotRunning,
        updateResult: '',
        version: rigProject.manifest.version,
      });
    }
  }

  public onChange = (input: React.FormEvent<HTMLInputElement>) => {
    const { checked, name, type, value } = input.currentTarget;
    if (name !== 'name') {
      if (name === 'version') {
        this.setState({ updateResult: '', version: input.currentTarget.value });
      } else {
        this.props.onChange({ [name]: type === 'checkbox' ? checked : value } as RigProject);
        setHttpOption(checked);
      }
    }
  }

  private dismissRefreshNotice = () => {
    const localStorage = new LocalStorage();
    localStorage.isRefreshAware = true;
    this.setState({ isRefreshAware: true });
  }

  private openProjectFolder = () => {
    openUrl(this.props.rigProject.projectFolderPath);
  }

  private selectBackendFolder = () => {
    this.selectFolder('backendFolderName');
  }

  private selectFrontendFolder = () => {
    this.selectFolder('frontendFolderName');
  }

  private selectFolder = async (name: string) => {
    let folderPath = await showOpenDialog({ properties: ['createDirectory ', 'openDirectory'] });
    if (folderPath) {
      const { projectFolderPath } = this.props.rigProject;
      if (folderPath.startsWith(projectFolderPath)) {
        folderPath = folderPath.substr(projectFolderPath.length + 1);
        folderPath = folderPath || projectFolderPath;
      }
      this.props.onChange({ [name]: folderPath } as any as RigProject);
    }
  }

  private toggleBackend = async () => {
    const backendState = this.state.backendResult;
    this.setState({ backendResult: HostingResult.None });
    const backendResult = await toggleBackend(this.props.rigProject, backendState);
    if (backendResult) {
      this.setState({ backendResult });
      this.props.refreshViews();
    }
  }

  private toggleFrontend = async () => {
    const frontendState = this.state.frontendResult;
    this.setState({ frontendResult: HostingResult.None });
    const frontendResult = await toggleFrontend(this.props.rigProject, frontendState);
    if (frontendResult) {
      this.setState({ frontendResult });
      this.props.refreshViews();
    }
  }

  private getExtensionViews(rigProject: RigProject): string {
    const extensionViewTypes = ['panel', 'component', 'videoOverlay', 'mobile'];
    return ['Panel', 'Component', 'Video Overlay', 'Mobile'].filter((_, index) => {
      return Object.getOwnPropertyDescriptor(rigProject.manifest.views, extensionViewTypes[index]);
    }).join(', ');
  }

  private getIsRunning(result: string) {
    return result === HostingResult.Started || result === HostingResult.Running;
  }

  private refreshManifest = async () => {
    const { manifest: { id: clientId, version } } = this.props.rigProject;
    const manifest = await fetchExtensionManifest(clientId, version);
    const secret = await fetchExtensionSecret(clientId);
    this.setState({ version: manifest.version });
    this.props.onChange({ secret, manifest } as RigProject);
  }

  private updateByVersion = async () => {
    this.setState({ updateResult: 'fetching...' });
    try {
      const { manifest: { id: clientId } } = this.props.rigProject;
      const manifest = await fetchExtensionManifest(clientId, this.state.version);
      this.setState({ updateResult: '' });
      this.props.onChange({ manifest } as RigProject);
    } catch (ex) {
      this.setState({ updateResult: 'not found' });
    }
  }

  static Property({ name, value }: PropertyProps) {
    return <>
      <FormGroup label={name}>
        <CoreText color={Color.Base}>{value}</CoreText>
      </FormGroup>
    </>;
  }

  static InputProperty({ action, disabled, hint, isDefault, label, name, placeholder, required, value, onChange }: InputPropertyProps) {
    return <>
      <FormGroup hint={hint} label={label} required={required}>
        <Layout display={Display.Flex} justifyContent={JustifyContent.End}>
          <Layout fullWidth margin={{ right: 1 }}>
            <Input disabled={disabled} name={name} placeholder={placeholder} type={InputType.Text} value={value} onChange={onChange} />
          </Layout>
          <Layout className="project-view__button">
            <Button ariaLabel={`ProjectView:${action.text}`} disabled={action.disabled} fullWidth type={isDefault ? ButtonType.Default : ButtonType.Hollow} onClick={action.onClick}>{action.text}</Button>
          </Layout>
          <Layout className="project-view__result" margin={{ left: 1 }}>
            <CoreText color={Color.Base}>{action.result}</CoreText>
          </Layout>
        </Layout>
      </FormGroup>
    </>;
  }

  private launchVisualStudioCode = () => {
    launchVisualStudioCode(this.props.rigProject.projectFolderPath);
  }

  public render() {
    const { rigProject, closeProject } = this.props;
    const { backendResult, frontendResult, hasVisualStudioCode, isRefreshAware, updateResult, version } = this.state;
    const isRemoteFrontEnd = !rigProject.frontendCommand && !rigProject.frontendFolderName;
    const isFrontendRunning = this.getIsRunning(frontendResult);
    const frontendCommandText = isFrontendRunning ? 'Stop Front End' : 'Run Front End';
    const isBackendRunning = this.getIsRunning(backendResult);
    const backendText = isBackendRunning ? 'Stop Back End' : 'Run Back End';
    const frontendFolderHintText = 'You may enter either an absolute path or a path relative to the project directory.  If your' +
      ' front end is no more than a collection of HTML, CSS, and JS files, such as for the Hello World example Extension, enter' +
      ' its path here and click "Run Front End".';
    const frontendCommandHintText = 'If your front end requires a custom command, such as for a React application, enter its' +
      ' activation command here and click "Run Front End".  The directory given in the previous text box or, if blank, the' +
      ' project directory is used as the command\'s working directory.';
    const portHintText = 'If this is checked, the Developer Rig will host your front end on a random port.  If it is not' +
      ' checked, the Developer Rig will use the port you specified in Testing Base URI.  If you do not explicitly specify a port' +
      ' in Testing Base URI, the Developer Rig will host your front end on a random port.  Note that Developer Rig hosting is' +
      ' always unencrypted (i.e., http instead of https).';
    const url = `https://dev.twitch.tv/console/extensions/${rigProject.manifest.id}/${rigProject.manifest.version}/status`;
    return <>
      <StyledLayout className="project-view" background={Background.Alt} fullHeight padding={3}>
        <Layout display={Display.Flex} fullWidth justifyContent={JustifyContent.Between}>
          <CoreText bold color={Color.Base} type={TextType.H3}>{NavItem.ProjectOverview}</CoreText>
          <Layout className="project-view__header" position={Position.Relative}>
            {isRefreshAware || <Balloon direction={BalloonDirection.Left} show={true} size={BalloonSize.Small}>
              <Layout margin={1}>
                <CoreText>Whenever you make a change to your extension
                  at <CoreLink ariaLabel="ProjectView:dev.twitch.tv" linkTo={url}>dev.twitch.tv</CoreLink>,
                  click this button to ensure the Developer Rig is up to date.</CoreText>
              </Layout>
              <Layout display={Display.Flex} justifyContent={JustifyContent.End} padding={1}>
                <Button ariaLabel="ProjectView:Got It" type={ButtonType.Default} onClick={this.dismissRefreshNotice}>Got It</Button>
              </Layout>
            </Balloon>}
            <Button ariaLabel="ProjectView:Refresh Manifest" icon={SVGAsset.Refresh} type={ButtonType.Default} onClick={this.refreshManifest}>Refresh Manifest</Button>
            <Button ariaLabel="ProjectView:Close Project" icon={SVGAsset.Hide} type={ButtonType.Hollow} onClick={closeProject}>Close Project</Button>
          </Layout>
        </Layout>
        <div>
          <StyledLayout className="project-view__section" background={Background.Base} border borderRadius={BorderRadius.Medium} padding={2}>
            <ProjectView.Property name="Extension Name" value={rigProject.manifest.name} />
            <ProjectView.Property name="Client ID" value={rigProject.manifest.id} />
            <ProjectView.Property name="Extension Types" value={this.getExtensionViews(rigProject)} />
            {hasVisualStudioCode &&
              <div>
                <Button type={ButtonType.Hollow} onClick={this.launchVisualStudioCode}>Open in VS Code</Button>
              </div>}
            <ProjectView.InputProperty action={{ text: 'Open', onClick: this.openProjectFolder }} disabled
              hint="This is the folder containing your project.  Click &quot;Open&quot; to view the files."
              label="Project Folder" name="projectFolderPath" value={rigProject.projectFolderPath} />
            <ProjectView.InputProperty
              action={{ disabled: !version, result: updateResult, text: 'Update', onClick: this.updateByVersion }}
              hint="Update the version of your extension project." label="Version" name="version"
              placeholder="Enter the version of your extension" required value={version} onChange={this.onChange} />
          </StyledLayout>
        </div>
        <CoreText bold color={Color.Base} type={TextType.H4}>How to run your Extension in the Developer Rig</CoreText>
        <div>
          <StyledLayout className="project-view__section" background={Background.Base} border borderRadius={BorderRadius.Medium} padding={2}>
            <CoreText className="project-view__text">To run your extension you'll need to host your front-end assets and, if you
              have a back end, run your back end.  If you would like the Rig to host your front end for you, enter the path to your
              front-end assets in the first text box below.  Otherwise, ensure your Base URI is correct in the Twitch Developer
              Console for your extension and either enter the command to host your front end in the second text box or leave both
              text boxes blank if you will host your front end yourself.</CoreText>
            {rigProject.manifest.state === 'Testing' ? <>
              <ProjectView.InputProperty
                action={{ disabled: isFrontendRunning, text: 'Select', onClick: this.selectFrontendFolder }}
                disabled={isFrontendRunning} hint={frontendFolderHintText} isDefault={!rigProject.frontendFolderName}
                label="Host your front-end files" name="frontendFolderName"
                placeholder="Enter the path to your front-end files" value={rigProject.frontendFolderName}
                onChange={this.onChange} />
              <div>
                <ProjectView.InputProperty
                  action={{ disabled: isRemoteFrontEnd, result: frontendResult, text: frontendCommandText, onClick: this.toggleFrontend }}
                  disabled={isFrontendRunning} hint={frontendCommandHintText} isDefault={!!rigProject.frontendFolderName}
                  label="Front-end Host Command" name="frontendCommand"
                  placeholder="Enter the command to run your front-end hosting" value={rigProject.frontendCommand}
                  onChange={this.onChange} />
                <FormGroup label="" hint={portHintText}>
                  <CheckBox name="usingRandomFrontendHostingPort" checked={rigProject.usingRandomFrontendHostingPort && !rigProject.frontendCommand}
                    disabled={isFrontendRunning || Boolean(rigProject.frontendCommand)} label="Use random port for hosting" onChange={this.onChange} />
                </FormGroup>
              </div>
            </> : <>
              <FormGroup label="Twitch is hosting your front-end files.">
                <Layout display={Display.Flex} justifyContent={JustifyContent.End}>
                  <Layout fullWidth margin={{ right: 1 }}>
                    <CoreText className="project-view__text">Your project is in the "{rigProject.manifest.state}" state.  The
                      Developer Rig will not host your front-end files.  Twitch does not allow your front end to access non-SSL
                      back-end services.  Check the following box to override this.  This applies to the Developer Rig only.</CoreText>
                  </Layout>
                  <Layout className="project-view__result" margin={{ left: 1 }} />
                </Layout>
              </FormGroup>
              <CheckBox name="allowHttpBackend" checked={rigProject.allowHttpBackend} label="Allow non-SSL back-end access" onChange={this.onChange} />
            </>}
          </StyledLayout>
        </div>
        <div>
          <StyledLayout className="project-view__section" background={Background.Base} border borderRadius={BorderRadius.Medium} padding={2}>
            <CoreText className="project-view__text">The Rig can invoke a process to run your extension back end locally.  Enter
              the path to the directory containing your back-end code and enter the command to invoke it.</CoreText>
            <ProjectView.InputProperty action={{ disabled: isBackendRunning, text: 'Select', onClick: this.selectBackendFolder }}
              disabled={isBackendRunning}
              hint="The path you enter here is used as the working directory of the back-end command.  You may enter either an absolute path or a path relative to the project directory."
              isDefault={!rigProject.backendFolderName} label="Back-end Files Location" name="backendFolderName"
              placeholder="Enter the path to your back-end files" value={rigProject.backendFolderName} onChange={this.onChange} />
            <ProjectView.InputProperty
              action={{ disabled: !rigProject.backendCommand, result: backendResult, text: backendText, onClick: this.toggleBackend }}
              disabled={isBackendRunning} hint="Enter the command to run your back-end service and click &quot;Run Back End&quot;." isDefault={!!rigProject.backendFolderName}
              label="Run your back-end service locally with the Developer Rig" name="backendCommand"
              placeholder="Your back-end command" value={rigProject.backendCommand} onChange={this.onChange} />
          </StyledLayout>
        </div>
      </StyledLayout>
    </>;
  }
}
