import * as React from 'react';
import classNames from 'classnames';
import './component.sass';
import whitetriangle from '../img/whitetriangle.svg';
import { sendEvent } from '../util/api';
import { Layout, Display } from 'twitch-core-ui';

export enum NavItem {
  ConfigurationService = 'Configuration Service',
  ExtensionViews = 'Extension Views',
  ProductManagement = 'Monetization',
  ProjectOverview = 'Project Details',
}

interface Props {
  currentView: NavItem;
  isOpen: boolean;
  onToggle: () => void;
  selectView: (navItem: NavItem) => void;
  showReleaseNotes: () => void;
}

export class RigNav extends React.Component<Props> {
  private selectView = (event: React.FormEvent<HTMLButtonElement>) => {
    const label = event.currentTarget.getAttribute('aria-label')!;
    sendEvent('dx_rig_nav_click', { tab_name: label });
    this.props.selectView(label.split(':')[1] as NavItem);
    event.stopPropagation();
  }

  public render() {
    const { currentView, isOpen, onToggle, showReleaseNotes } = this.props;
    const primaryItems = [
      {
        navItem: NavItem.ProjectOverview,
        svg: (
          <svg width="16" height="16" viewBox="0 0 16 16" xmlns="http://www.w3.org/2000/svg">
            <rect width="7" height="7" />
            <rect x="8" width="7" height="7" />
            <rect y="8" width="7" height="7" />
            <rect x="8" y="8" width="7" height="7" />
          </svg>
        ),
      },
      {
        navItem: NavItem.ExtensionViews,
        svg: (
          <svg width="16" height="16" viewBox="0 0 16 16" xmlns="http://www.w3.org/2000/svg">
            <path d="M14 7H13V5C13 3.896 12.104 3 11 3H9V2C9 0.896 8.104 0 7 0C5.896 0 5 0.896 5 2V3H2C0.896 3 0 3.896 0 5V7H1C2.104 7 3 7.896 3 9C3 10.104 2.104 11 1 11H0V14C0 15.104 0.896 16 2 16H5V14.937C5 13.833 5.896 12.937 7 12.937C8.104 12.937 9 13.833 9 14.937V16H11C12.104 16 13 15.104 13 14V11H14C15.104 11 16 10.104 16 9C16 7.896 15.104 7 14 7Z" />
          </svg>
        ),
      },
      {
        navItem: NavItem.ProductManagement,
        svg: (
          <svg width="14" height="16" viewBox="0 0 14 16" xmlns="http://www.w3.org/2000/svg">
            <path d="M7.40751 7.08112C7.14951 6.97412 6.85051 6.97412 6.59151 7.08112L-0.000488281 9.80212L6.12151 0.466122C6.31251 0.176122 6.64351 0.00012207 6.99951 0.00012207C7.35651 0.00012207 7.68751 0.176122 7.87751 0.466122L13.9995 9.80212L7.40751 7.08112ZM0.999712 11.638L6.99971 9.00002L12.9997 11.638L7.59271 15.798C7.41671 15.933 7.20771 16 6.99971 16C6.79171 16 6.58271 15.933 6.40671 15.798L0.999712 11.638Z" />
          </svg>
        ),
      },
      {
        navItem: NavItem.ConfigurationService,
        svg: (
          <svg width="16" height="16" viewBox="0 0 16 16" xmlns="http://www.w3.org/2000/svg">
            <path d="M7.97051 11.3945C6.10951 11.3945 4.60051 9.88551 4.60051 8.02451C4.60051 6.16351 6.10951 4.65451 7.97051 4.65451C9.83151 4.65451 11.3405 6.16351 11.3405 8.02451C11.3405 9.88551 9.83151 11.3945 7.97051 11.3945ZM15.2705 6.47951C14.9695 6.43451 14.6535 6.39151 14.3445 6.35151C14.3075 6.34651 14.2735 6.32151 14.2635 6.28751C14.1205 5.77051 13.9125 5.27251 13.6485 4.80651C13.6305 4.77551 13.6325 4.73651 13.6555 4.70651C13.8455 4.45951 14.0325 4.21051 14.2135 3.96651C14.4575 3.63551 14.4215 3.16651 14.1305 2.87551L13.1195 1.86451C12.8275 1.57351 12.3595 1.53751 12.0295 1.78251C11.7845 1.96251 11.5355 2.15051 11.2885 2.34051C11.2585 2.36351 11.2195 2.36651 11.1885 2.34951C10.7225 2.08351 10.2245 1.87751 9.70751 1.73551C9.67351 1.72651 9.64851 1.69551 9.64351 1.65851C9.60351 1.34951 9.55951 1.02751 9.51451 0.727512C9.45351 0.320512 9.09751 -0.000488281 8.68551 -0.000488281H7.25551C6.84351 -0.000488281 6.48751 0.320512 6.42651 0.727512C6.38151 1.02751 6.33751 1.34251 6.29751 1.65151C6.29251 1.68851 6.26751 1.72251 6.23351 1.73151C5.71751 1.87451 5.21851 2.08251 4.75251 2.34751C4.72151 2.36451 4.68251 2.36251 4.65251 2.33951C4.40551 2.14951 4.15651 1.96251 3.91251 1.78151C3.58151 1.53751 3.11351 1.57251 2.82151 1.86451L1.81051 2.87551C1.51951 3.16651 1.48351 3.63551 1.72851 3.96651C1.90951 4.21051 2.09751 4.45951 2.28651 4.70651C2.30951 4.73651 2.31251 4.77551 2.29551 4.80651C2.03051 5.27251 1.82451 5.77051 1.68151 6.28751C1.67251 6.32151 1.64151 6.34651 1.60451 6.35151C1.29651 6.39151 1.00051 6.43451 0.699512 6.47951C0.293512 6.54051 -0.000488281 6.89751 -0.000488281 7.30951V8.73951C-0.000488281 9.15151 0.293512 9.50751 0.699512 9.56851C1.00051 9.61351 1.30351 9.65751 1.61151 9.69751C1.64851 9.70251 1.67551 9.72751 1.68451 9.76151C1.82651 10.2775 2.03251 10.7755 2.29651 11.2425C2.31451 11.2735 2.31051 11.3125 2.28751 11.3425C2.09751 11.5885 1.90951 11.8375 1.72851 12.0825C1.48451 12.4135 1.51951 12.8815 1.81051 13.1725L2.82151 14.1845C3.11351 14.4755 3.58151 14.5105 3.91251 14.2665C4.15751 14.0855 4.40551 13.8985 4.65251 13.7075C4.68251 13.6855 4.72151 13.6815 4.75251 13.7005C5.21851 13.9645 5.71651 14.1705 6.23351 14.3135C6.26751 14.3225 6.29251 14.3525 6.29751 14.3905C6.33751 14.6995 6.38151 14.9975 6.42651 15.2975C6.48751 15.7045 6.84351 15.9995 7.25551 15.9995H8.68551C9.09751 15.9995 9.45351 15.7045 9.51451 15.2965C9.55951 14.9975 9.60351 14.6935 9.64351 14.3845C9.64851 14.3475 9.67351 14.3205 9.70751 14.3105C10.2245 14.1675 10.7225 13.9625 11.1885 13.6985C11.2195 13.6805 11.2585 13.6845 11.2885 13.7075C11.5355 13.8975 11.7845 14.0855 12.0295 14.2665C12.3595 14.5105 12.8275 14.4755 13.1185 14.1835L14.1305 13.1725C14.4215 12.8815 14.4565 12.4135 14.2125 12.0815C14.0325 11.8375 13.8435 11.5885 13.6545 11.3425C13.6315 11.3125 13.6275 11.2735 13.6465 11.2425C13.9105 10.7755 14.1165 10.2775 14.2595 9.76151C14.2685 9.72751 14.2985 9.70251 14.3365 9.69751C14.6445 9.65751 14.9695 9.61351 15.2705 9.56851C15.6765 9.50751 15.9995 9.15151 15.9995 8.73951V7.30951C15.9995 6.89751 15.6765 6.54051 15.2705 6.47951Z" />
          </svg>
        ),
      },
    ];
    const secondaryItems = [
      {
        svg: (
          <svg width="14" height="16" viewBox="0 0 14 16" xmlns="http://www.w3.org/2000/svg">
            <path fillRule="evenodd" d="M8.333 10.097C7.597 10.804 9.619 13.928 6.99 14H2V2H12V6.997C12 7.003 12.003 7.008 12.003 7.014V9.019C12.003 11.116 9.07 9.518 8.333 10.097ZM14 1V1.002V1.003V6.997C14 7.003 13.997 7.008 13.997 7.014V9.591C13.997 11.185 9.564 16 7.456 16H6.997H2.003H1.003H1.001H1C0.771 16 0.569 15.909 0.4 15.78C0.366 15.753 0.324 15.738 0.293 15.706C0.282 15.696 0.277 15.679 0.267 15.667C0.105 15.49 0 15.259 0 15V14.999C0 14.998 0 14.998 0 14.997V1.003V1.002C0 1.001 0 1.001 0 1C0 0.773 0.089 0.575 0.217 0.407C0.245 0.37 0.261 0.326 0.293 0.293C0.301 0.286 0.312 0.283 0.32 0.276C0.499 0.108 0.735 -4.97818e-08 1 -4.97818e-08L1.001 0.000999997C1.002 0.000999997 1.002 -4.97818e-08 1.003 -4.97818e-08H12.997C12.998 -4.97818e-08 12.998 0.000999997 12.999 0.000999997L13 -4.97818e-08C13.25 -4.97818e-08 13.472 0.101 13.647 0.253C13.666 0.269 13.689 0.277 13.707 0.293C13.73 0.317 13.741 0.35 13.763 0.376C13.904 0.549 14 0.76 14 1ZM5 6.0003H9C9.553 6.0003 10 5.5533 10 5.0003C10 4.4473 9.553 4.0003 9 4.0003H5C4.448 4.0003 4 4.4473 4 5.0003C4 5.5533 4.448 6.0003 5 6.0003ZM5 9.0003H9C9.553 9.0003 10 8.5533 10 8.0003C10 7.4473 9.553 7.0003 9 7.0003H5C4.448 7.0003 4 7.4473 4 8.0003C4 8.5533 4.448 9.0003 5 9.0003Z" />
          </svg>
        ),
        text: 'Documentation',
        url: 'https://dev.twitch.tv/docs/extensions/',
      },
      {
        svg: (
          <svg width="16" height="16" viewBox="0 0 46.5 53.5" xmlns="http://www.w3.org/2000/svg">
            <g strokeWidth="0.26458332" fill="white" transform="translate(0,-243)">
              <path d="m 18.362083,266.28188 c -1.508125,0 -2.69875,1.32291 -2.69875,2.93687 0,1.61396 1.217084,2.93688 2.69875,2.93688 1.508125,0 2.69875,-1.32292 2.69875,-2.93688 0.02646,-1.61396 -1.190625,-2.93687 -2.69875,-2.93687 z m 9.657292,0 c -1.508125,0 -2.69875,1.32291 -2.69875,2.93687 0,1.61396 1.217083,2.93688 2.69875,2.93688 1.508121,0 2.698751,-1.32292 2.698751,-2.93688 0,-1.61396 -1.19063,-2.93687 -2.698751,-2.93687 z" />
              <path d="M 40.878126,244.08333 H 5.423958 C 2.434166,244.08333 0,246.5175 0,249.53375 v 35.77167 c 0,3.01625 2.434166,5.45041 5.423958,5.45041 h 30.003748 l -1.40229,-4.89479 3.38667,3.14854 3.20146,2.96334 5.68854,5.02708 v -47.46625 c 0,-3.01625 -2.43417,-5.45042 -5.42396,-5.45042 z m -10.21292,34.55459 c 0,0 -0.9525,-1.13771 -1.746248,-2.14313 3.466038,-0.97896 4.788958,-3.14854 4.788958,-3.14854 -1.08479,0.71438 -2.11667,1.21708 -3.04271,1.56104 -1.322914,0.55563 -2.592914,0.92604 -3.836456,1.13771 -2.54,0.47625 -4.868333,0.34396 -6.852708,-0.0265 -1.508125,-0.29104 -2.804584,-0.71437 -3.889375,-1.13771 -0.608542,-0.23812 -1.27,-0.52916 -1.931459,-0.89958 -0.07938,-0.0529 -0.15875,-0.0794 -0.238125,-0.13229 -0.05292,-0.0265 -0.07938,-0.0529 -0.105833,-0.0794 -0.47625,-0.26459 -0.740833,-0.44979 -0.740833,-0.44979 0,0 1.27,2.11666 4.630208,3.12208 -0.79375,1.00542 -1.772708,2.19604 -1.772708,2.19604 -5.847292,-0.18521 -8.069792,-4.02167 -8.069792,-4.02167 0,-8.51958 3.81,-15.4252 3.81,-15.4252 3.81,-2.8575 7.434792,-2.77813 7.434792,-2.77813 l 0.264583,0.3175 c -4.7625,1.37583 -6.958542,3.46604 -6.958542,3.46604 0,0 0.582084,-0.3175 1.561042,-0.76729 2.831042,-1.24354 5.08,-1.5875 6.006042,-1.66687 0.15875,-0.0265 0.291041,-0.0529 0.449791,-0.0529 1.613959,-0.21167 3.439584,-0.26459 5.344584,-0.0529 2.513541,0.29104 5.212289,1.03187 7.963959,2.54 0,0 -2.09021,-1.98438 -6.588126,-3.36021 l 0.370417,-0.42334 c 0,0 3.624789,-0.0794 7.434789,2.77813 0,0 3.81,6.90562 3.81,15.42521 0,0 -2.24896,3.83646 -8.09625,4.02166 z" />
            </g>
          </svg>
        ),
        text: 'Join us on Discord',
        url: 'https://discord.gg/ZFC9AZy',
      },
    ];
    const textClassName = classNames('rig-nav__text', {
      'tw-hidden': !isOpen,
    });
    const toggleClassName = classNames('rig-nav__toggle', {
      'rig-nav__toggle--icon': !isOpen,
    });
    const triangleClass = classNames('rig-nav__triangle', {
      'rig-nav__triangle--open': isOpen,
    });
    return <>
      <Layout display={Display.Flex}>
        <div className="rig-nav__items">
          {primaryItems.map((item) => {
            const { navItem, svg } = item;
            const linkClassName = classNames('rig-nav__link', {
              'rig-nav__link--selected': navItem === currentView,
            });
            const title = isOpen ? undefined : navItem;
            return <div key={navItem} className="rig-nav__item">
              <button aria-label={`RigNav:${navItem}`} className={linkClassName} title={title} onClick={this.selectView}>
                <div className={linkClassName}>
                  {svg}
                  <div className={textClassName}>{navItem}</div>
                </div>
              </button>
            </div>;
          })}
          <div className="rig-nav__item">
            <span className="rig-nav__link rig-nav__link--separator">
              <div className="rig-nav__no-image" />
              <div className={textClassName}>Need Help?</div>
            </span>
          </div>
          {secondaryItems.map((item) => {
            const { svg, text, url } = item;
            return <div key={text} className="rig-nav__item">
              <a aria-label={`RigNav:${text}`} className="rig-nav__link" href={url}>
                <div className="rig-nav__image" title={isOpen ? undefined : text}>{svg}</div>
                <div className={textClassName}>{isOpen ? text : undefined}</div>
              </a>
            </div>;
          })}
          <div className="rig-nav__item">
            <button aria-label="RigNav:Release Notes" className="rig-nav__link" title={isOpen ? undefined : 'Release Notes'} onClick={showReleaseNotes}>
              <div className="rig-nav__link">
                <svg width="16" height="16" viewBox="0 0 256 256" xmlns="http://www.w3.org/2000/svg">
                  <g transform="matrix(2.8111682,0,0,2.8111682,43.648931,0.59369665)">
                    <g transform="translate(-95,-397)">
                      <path d="m 148.86914,416.38956 2.89582,2.46698 c 3.15063,2.60426 3.97857,7.10482 1.99049,10.58541 l -11.57901,21.01376 -0.1174,0.18074 c -1.57849,2.43025 -4.25989,3.92056 -7.15667,3.98263 l -6.62405,3e-4 c 0.24068,0.78595 0.82895,1.83232 2.1216,3.12858 1.8715,1.871 11.13709,7.1055 11.141,15.461 0,8.352 -6.3715,14.434 -16.266,14.434 -9.8948,0 -13.26203,-8.2074 -13.555,-12.016 -0.22652,-2.9375 -0.73433,-5.3555 -1.0273,-7.2578 -0.293,-1.9102 -0.7305,-3.81253 1.2461,-4.1055 1.9766,-0.293 3.875,1.6875 4.9805,5.0586 0.8593,2.6211 2.4687,6.01951 5.7148,6.8906 2.4844,0.6641 6.96096,0.3633 7.5469,-4.3242 0.586,-4.6875 -4.5117,-7.4414 -6.7148,-9.6445 -1.97055,-1.96679 -1.97625,-5.86399 -1.86608,-7.62447 l -6.37509,2.8e-4 c -3.0239,-0.0633 -5.70533,-1.55367 -7.28389,-3.98381 l -0.11746,-0.18083 -0.10406,-0.18886 -11.430926,-20.74642 c -2.031736,-3.55866 -1.20419,-8.05981 1.946503,-10.66406 L 101,416.50074 V 399.5 c 0,-1.38071 1.11929,-2.5 2.5,-2.5 1.38071,0 2.5,1.11929 2.5,2.5 v 12.74151 l 3.64773,-3.10731 c 1.57204,-1.35059 3.58651,-2.07312 5.65558,-2.0312 h 19.3936 c 2.06374,-0.0418 4.07313,0.67688 5.64338,2.02074 l 3.52885,3.00626 V 399.5 c 0,-1.38071 1.11929,-2.5 2.5,-2.5 1.38071,0 2.5,1.11929 2.5,2.5 z m -12.39403,-2.74222 c -0.47149,-0.40508 -1.07907,-0.61758 -1.7,-0.59434 h -19.55 c -0.6209,-0.0232 -1.2285,0.18926 -1.7,0.59434 l -11.475,9.775 c -0.94299,0.76035 -1.19527,2.09508 -0.59433,3.14763 l 11.47499,20.825 c 0.50469,0.77696 1.36468,1.25511 2.29432,1.27501 l 19.5498,4.7e-4 c 0.92973,-0.0199 1.78968,-0.49805 2.29432,-1.275 l 11.475,-20.825 c 0.60098,-1.05255 0.34864,-2.38731 -0.59434,-3.14763 z m -1.06931,10.42339 4.77765,4.8762 c 0.74883,0.7524 0.74883,1.87738 0,2.6262 l -4.78125,4.78125 c -0.3726,0.37618 -0.84372,0.5625 -1.3113,0.5625 -0.46755,0 -0.93513,-0.18632 -1.3113,-0.5625 -0.74883,-0.7524 -0.74883,-1.87737 0,-2.6262 l 3.46995,-3.46995 -3.46995,-3.5613 c -0.74883,-0.7524 -0.74883,-1.87737 0,-2.6262 0.7524,-0.74882 1.87737,-0.74882 2.6262,0 z m -6.8445,-4.968 -6e-5,-2e-5 c 0.93871,0.28125 1.50121,1.31139 1.21644,2.34495 l -5.2488,18.657 c -0.1899,0.84375 -0.9387,1.40625 -1.78245,1.40625 -0.0914,0 -0.28125,0 -0.46758,-0.0949 -1.03005,-0.28125 -1.59255,-1.3149 -1.3113,-2.34495 l 5.2488,-18.657 c 0.28125,-1.03005 1.3149,-1.59255 2.34495,-1.3113 z m -11.3418,4.968 h -9e-5 c 0.74883,0.7488 0.74883,1.96873 0,2.71755 l -3.46995,3.46995 3.46644,3.375 c 0.74883,0.7524 0.74883,1.87738 0,2.6262 -0.3726,0.37618 -0.84375,0.5625 -1.3113,0.5625 -0.46755,0 -0.93513,-0.18632 -1.3113,-0.5625 l -4.78125,-4.78125 c -0.74883,-0.7524 -0.74883,-1.87737 0,-2.6262 l 4.78125,-4.78125 c 0.7524,-0.74882 1.87737,-0.74882 2.6262,0 z"></path>
                    </g>
                  </g>
                </svg>
                <div className={textClassName}>Release Notes</div>
              </div>
            </button>
          </div>
        </div>
        <div className={toggleClassName} onClick={onToggle}>
          <div className="rig-nav__toggle-indicator">
            <img alt="^" src={whitetriangle} className={triangleClass} />
          </div>
        </div>
      </Layout>
    </>;
  }
}
