import { setupShallowTest } from '../tests/enzyme-util/shallow';
import { createViewsForTest, createExtensionManifestForTest } from '../tests/constants/extension';
import { Rig } from './component';
import { RigExtensionView, RigProject, RigProjectReference, ViewerType } from '../core/models/rig';
import { ExtensionAnchor } from 'extension-coordinator';
import { LocalStorage } from '../util/local-storage';
import { timeout } from '../tests/utils';
import { ExtensionManifest } from '../core/models/manifest';
import { UserSession } from '../core/models/user-session';

const localStorage = new LocalStorage();
const userSession: UserSession = {
  authToken: 'authToken',
  displayName: 'displayName',
  email: 'email@email.com',
  expiration: Date.parse('3333-03-03'),
  id: 'id',
  login: 'login',
  profileImageUrl: 'profileImageUrl',
};
localStorage.rigLogin = userSession;

let globalAny = global as any;

const setupShallow = setupShallowTest(Rig, () => ({}));

function mockApiFunctions() {
  const rigProjectTemplate = {
    certificateExceptions: [],
    manifest: {},
  };
  return {
    ...require.requireActual('../util/api'),
    checkForUpdate: jest.fn(() => Promise.resolve({})),
    fetchChannelConfigurationSegments: jest.fn(() => Promise.resolve({})),
    fetchGlobalConfigurationSegment: jest.fn(() => Promise.resolve({})),
    loadFile: jest.fn().mockImplementation((path) => {
      const rigProject = Object.assign({}, rigProjectTemplate, { certificateExceptions: [], extensionViews: [], manifest: { id: path } });
      return Promise.resolve(JSON.stringify(rigProject));
    }),
    showOpenDialog: jest.fn(),
  };
}
jest.mock('../util/api', () => mockApiFunctions());
const api = require.requireMock('../util/api');

describe('<Rig />', () => {
  function setUpProjectForTest(): RigProjectReference[] {
    const [filePath, name, secret] = ['filePath', 'name', 'secret'];
    const projectReferences = [
      { filePath: filePath + '0', name, secret },
      { filePath: filePath + '1', name, secret },
    ] as RigProjectReference[];
    localStorage.projectReferences = projectReferences;
    localStorage.currentProjectIndex = 0;
    return projectReferences;
  }

  function createProject(manifest: ExtensionManifest): RigProject {
    return { certificateExceptions: [], extensionViews: [], manifest } as unknown as RigProject;
  }

  it('renders correctly', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout(111);
    expect(wrapper.debug()).toMatchSnapshot();
  });

  it('gets project references from local storage correctly', async () => {
    const projectReferences = setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout();
    expect(wrapper.state('projectReferences')).toEqual(projectReferences);
  });

  it('correctly toggles state when create extension view is opened/closed', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout();
    const instance = wrapper.instance() as any;
    instance.state.currentProject = createProject(createExtensionManifestForTest());
    const expectedView = { channelId: 'developerrig' } as RigExtensionView;
    await instance.createExtensionView(expectedView);
    expect(wrapper.state<{ extensionViews: RigExtensionView[] }>('currentProject').extensionViews[0]).toBe(expectedView);
  });

  globalAny.confirm = jest.fn().mockImplementation(() => true);
  globalAny.fetch = jest.fn().mockImplementation(() => Promise.resolve({
    status: 200,
    json: () => Promise.resolve({}),
  }));

  it('adds project', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout();
    const instance = wrapper.instance() as any;
    const extensionViews = createViewsForTest(1, ExtensionAnchor.Panel, ViewerType.LoggedOut);
    const project = createProject({} as ExtensionManifest);
    project.extensionViews = extensionViews;
    await instance.addProject(project, 'projectFilePath', 'secret');
    expect(wrapper.state('showingCreateProjectDialog')).toBe(false);
    expect(wrapper.state('openingProject')).not.toBeDefined();
  });

  it('selects project', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout();
    const instance = wrapper.instance() as any;
    await instance.selectProject(1);
    expect(wrapper.state<{ manifest: { id: string } }>('currentProject').manifest.id).toBe('filePath1');
  });

  it('deletes project', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout();
    const currentProject = createProject(createExtensionManifestForTest());
    wrapper.setState({ currentProject });
    const instance = wrapper.instance() as any;
    await instance.closeProject();
    expect(globalAny.confirm).toHaveBeenCalled();
    expect(wrapper.state('currentProject')).not.toBe(currentProject);
  });

  it('renders and closes CreateProjectWizard', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout(11);
    const instance = wrapper.instance() as any;
    instance.showCreateProjectDialog();
    expect(wrapper.find('CreateProjectWizard').length).toBe(1);
    instance.closeProjectDialog();
    await timeout();
    expect(wrapper.find('CreateProjectWizard').length).toBe(0);
  });

  it('set user session', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout();
    const instance = wrapper.instance() as any;
    await instance.setUserSession(userSession);
    expect(wrapper.state('session')).toBe(userSession);
  });

  it('handles main messages', async () => {
    setUpProjectForTest();
    const { wrapper } = setupShallow();
    await timeout();
    const instance = wrapper.instance() as any;
    instance.handleMainMessage(undefined, { action: 'create project' });
    expect(wrapper.state('showingCreateProjectDialog')).toBe(true);
    instance.handleMainMessage(undefined, { action: 'open project' });
    expect(api.showOpenDialog).toHaveBeenCalled();
    let warn = console.warn;
    console.warn = jest.fn();
    instance.handleMainMessage(undefined, { action: 'unknown' });
    [console.warn, warn] = [warn, console.warn];
    expect(warn).toHaveBeenCalledWith('unexpected main message "unknown"');
  });
});
