import * as React from 'react';
import classNames from 'classnames';
import { RigNav } from '../rig-nav';
import { ExtensionViewContainer } from '../extension-view-container';
import { ProductsView } from '../products-view';
import { checkForUpdate, loadFile, showContextMenu, showMessageBox, showOpenDialog, stopHosting, viewInBrowser, sendEvent, addCertificateException, clearAuth, setProject } from '../util/api';
import { NavItem } from '../rig-nav'
import { RigExtensionView, RigProject, RigProjectReference, RigContext, parseRigProject } from '../core/models/rig';
import { UserSession } from '../core/models/user-session';
import { WelcomePage } from '../welcome-page';
import { ProjectView } from '../project-view';
import { ConfigurationServiceView } from '../configuration-service-view';
import { LocalStorage } from '../util/local-storage';
import { ProjectNav } from '../project-nav';
import electron from '../electron';
import { OpenProjectDialog } from '../open-project-dialog';
import { DropdownMenu } from '../dropdown-menu';
import { ProjectUpdate } from '../util/project-update';
import { LoadFailureView } from '../load-failure-view';
import { ReleaseNotesDialog } from '../release-notes-dialog';
import { CreateProjectWizard } from '../create-project-wizard';
import { initGraphQLClient } from '../util/graphql-client';
import './component.sass';
import { Button, ButtonType, StyledLayout, Background } from 'twitch-core-ui';
import { CertificateExceptionRequest, CertificateException } from '../core/models/certificate-exception';
import { CheckCertificateDialog } from '../check-certificate-dialog';

const localStorage = new LocalStorage();

interface OpeningProject {
  currentProject: RigProject;
  filePath: string;
}

type Props = {};

class State {
  currentProject?: RigProject;
  currentProjectIndex: number;
  currentView: NavItem;
  exceptionRequest?: CertificateExceptionRequest;
  extensionsViewContainerKey: number;
  isReady: boolean;
  isRigNavOpen: boolean;
  isUpToDate?: boolean;
  openingProject?: OpeningProject;
  projectReferences: RigProjectReference[];
  rigContext?: React.ContextType<typeof RigContext>;
  session?: UserSession;
  showingCreateProjectDialog: boolean;
  showingLoadFailureView: boolean;
  showingReleaseNotes: boolean;
  themeClass: string;

  constructor() {
    this.currentProjectIndex = localStorage.currentProjectIndex;
    this.currentView = NavItem.ProjectOverview;
    this.extensionsViewContainerKey = 0;
    this.isReady = false;
    this.isRigNavOpen = true;
    this.projectReferences = localStorage.projectReferences;
    this.showingCreateProjectDialog = false;
    this.showingLoadFailureView = false;
    this.showingReleaseNotes = false;
    if (this.currentProjectIndex < 0 && this.projectReferences.length) {
      this.currentProjectIndex = 0;
    }
    this.session = localStorage.rigLogin;
    // hardcoded theme class until we make it user-selectable
    this.themeClass = 'tw-root--theme-light';
  }
}

export class Rig extends React.Component<Props, State> {
  public state: State = new State();
  private lastProjectFieldName: string = '';
  private projectUpdate: ProjectUpdate = new ProjectUpdate();
  private graphQLClient = initGraphQLClient();

  constructor(props: Readonly<Props>) {
    super(props);
    const { session } = this.state;
    if (session) {
      this.graphQLClient.setHeader('Authorization', `OAuth ${session.authToken}`);
    }
  }

  public async componentDidMount() {
    await checkForUpdate();
    this.setState({ isUpToDate: true });
    await this.loadProjects();
    if (this.state.currentProject) {
      this.setState({ rigContext: this.createRigContext() });
    }
    this.setState({ isReady: true });
    electron.ipcRenderer.on('main', this.handleMainMessage);
    window.addEventListener('contextmenu', Rig.onContextMenu, false);
  }

  public componentWillUnmount() {
    window.removeEventListener('contextmenu', Rig.onContextMenu, false);
    document.body.onclick = null;
    this.projectUpdate.stop();
  }

  private static onContextMenu(event: Event) {
    event.preventDefault();
    showContextMenu();
  }

  private get currentProject(): RigProject & RigProjectReference {
    return this.state.currentProject! && Object.assign({}, this.state.currentProject, this.state.projectReferences[this.state.currentProjectIndex]);
  }

  private get session() {
    return this.state.session!;
  }

  private get userId() {
    return this.session.id;
  }

  private createRigContext(project?: RigProject) {
    project = project || this.currentProject;
    const rigContext = {
      isConfigurationHosted: project.manifest.configurationLocation === 'hosted',
      secret: project.secret,
      userId: this.userId,
    };
    return rigContext;
  }

  private createExtensionView = async (extensionView: RigExtensionView) => {
    const extensionViews = this.currentProject.extensionViews || [];
    const nextExtensionViewId = 1 + extensionViews.reduce((reduced: number, view: RigExtensionView) => {
      return Math.max(reduced, parseInt(view.id, 10));
    }, 0);
    extensionView.id = nextExtensionViewId.toString();
    extensionViews.push(extensionView);
    this.updateExtensionViews(extensionViews);
  }

  private deleteExtensionView = (id: string) => {
    this.updateExtensionViews(this.currentProject.extensionViews.filter(element => element.id !== id));
  }

  private updateExtensionViews = (extensionViews: RigExtensionView[]) => {
    this.updateProject({ extensionViews } as RigProject);
  }

  private addProject = async (project: RigProject, filePath: string, secret: string) => {
    project.projectFolderPath = electron.dirname(filePath);
    this.state.projectReferences.length && await stopHosting();
    if (!project.version) {
      project.certificateExceptions = [];
      project.version = 1;
    }
    await setProject(this.userId, project);
    this.setState((previousState) => {
      const currentProjectIndex = previousState.projectReferences.length;
      localStorage.currentProjectIndex = currentProjectIndex;
      const projectReferences = [...previousState.projectReferences, { filePath, name: project.name, secret }];
      localStorage.projectReferences = projectReferences;
      return {
        currentProject: project,
        currentProjectIndex,
        currentView: NavItem.ProjectOverview,
        exceptionRequest: undefined,
        openingProject: undefined,
        projectReferences,
        rigContext: {
          isConfigurationHosted: project.manifest.configurationLocation === 'hosted',
          secret,
          userId: this.userId,
        },
        showingCreateProjectDialog: false,
      };
    });
    this.refreshViews();
  }

  private cancelOpeningProject = () => {
    this.setState({ openingProject: undefined });
  }

  private closeCheckCertificateDialog = async (certificateException: CertificateException) => {
    await addCertificateException(certificateException);
    if (!certificateException.isEphemeral) {
      const certificateExceptions = [...this.currentProject.certificateExceptions, certificateException];
      this.updateProject({ certificateExceptions } as RigProject);
    }
    this.setState({ exceptionRequest: undefined });
  }

  private closeProjectDialog = () => {
    this.setState({ showingCreateProjectDialog: false });
  }

  private closeProject = async () => {
    const message = `Are you sure you want to close project "${this.currentProject.name}"?\n\n` +
      'This will not delete any files or affect your extension online.';
    const isClosingProject = navigator.platform === 'Win32' ?
      await showMessageBox({ buttons: ['OK', 'Cancel'], message, type: 'question' }) === 0 :
      /* eslint no-restricted-globals: "off" */
      confirm(message);
    /* eslint no-restricted-globals: "error" */
    if (isClosingProject) {
      await stopHosting();
      await this.removeProject();
    }
  }

  private loadProject = async (projectReference: RigProjectReference) => {
    try {
      const { filePath, secret } = projectReference;
      const fileText = await loadFile(filePath);
      const currentProject = {
        ...parseRigProject(fileText),
        projectFolderPath: electron.dirname(filePath),
        secret,
      } as RigProject;
      if (!currentProject.version) {
        currentProject.certificateExceptions = [];
        currentProject.version = 1;
      }
      await setProject(this.userId, currentProject);
      sendEvent('dx_rig_load_project', { code_used: '' });
      this.setState({ currentProject, showingLoadFailureView: false });
      this.refreshViews();
    } catch (ex) {
      this.setState({ showingLoadFailureView: true });
    }
  }

  private loadProjects = async () => {
    if (this.state.session) {
      const { projectReferences } = this.state;
      if (projectReferences.length) {
        const projectReference = projectReferences[this.state.currentProjectIndex];
        this.setState({ currentView: NavItem.ProjectOverview });
        await this.loadProject(projectReference);
      }
    }
  }

  private openProject = async () => {
    const filePath = await showOpenDialog({
      filters: [
        { name: 'JSON files', extensions: ['json'] },
        { name: 'All Files', extensions: ['*'] },
      ],
      properties: ['openFile'],
    });
    if (filePath) {
      try {
        const fileText = await loadFile(filePath);
        this.setState({ exceptionRequest: undefined, showingCreateProjectDialog: false, showingLoadFailureView: false });
        const index = this.state.projectReferences.findIndex((reference) => reference.filePath === filePath);
        if (~index) {
          this.selectProject(index);
        } else {
          const currentProject = parseRigProject(fileText);
          if (this.state.currentProject && currentProject.manifest.id === this.state.currentProject.manifest.id) {
            this.addProject(currentProject, filePath, this.state.projectReferences[this.state.currentProjectIndex].secret);
          } else {
            this.setState({ openingProject: { currentProject, filePath } });
          }
        }
      } catch (ex) {
        showMessageBox({ title: 'Open Project Error', message: `Cannot open "${filePath}": ${ex.message}`, type: 'error' });
      }
    }
  }

  private selectProject = async (index: number) => {
    if (index !== this.state.currentProjectIndex) {
      this.lastProjectFieldName = '';
      await stopHosting();
      this.setState({ isReady: false, currentProjectIndex: index, currentView: NavItem.ProjectOverview });
      localStorage.currentProjectIndex = index;
      const projectReference = this.state.projectReferences[index];
      await this.loadProject(projectReference);
      this.setState({ isReady: true, rigContext: this.createRigContext() });
    }
  }

  private showCheckCertificateDialog = (exceptionRequest: CertificateExceptionRequest) => {
    this.setState({ exceptionRequest });
  }

  private showCreateProjectDialog = () => {
    this.setState({ showingCreateProjectDialog: true });
  }

  private updateProject = (project: RigProject) => {
    this.setState((previousState) => {
      const { filePath } = previousState.projectReferences[previousState.currentProjectIndex];
      const currentProject = Object.assign({}, previousState.currentProject, project);
      this.projectUpdate.setUpdate(filePath, currentProject);
      if (previousState.currentProject) {
        Object.keys(currentProject).forEach((name) => {
          if (this.lastProjectFieldName !== name && currentProject[name] !== previousState.currentProject![name]) {
            this.lastProjectFieldName = name;
            sendEvent('dx_rig_update_project', { update_label: name });
          }
        });
      }
      if (project.secret) {
        localStorage.projectReferences[previousState.currentProjectIndex].secret = project.secret;
      }
      if (project.manifest) {
        setProject(this.userId, currentProject);
        return { currentProject, rigContext: this.createRigContext(currentProject) };
      }
      return { currentProject };
    });
  }

  private logout = () => {
    clearAuth();
    localStorage.rigLogin = undefined;
    this.setState({ session: undefined });
  }

  private selectView = (navItem: NavItem) => {
    this.setState({ currentView: navItem });
  }

  private refreshViews = () => {
    this.setState((previousState) => ({ extensionsViewContainerKey: previousState.extensionsViewContainerKey + 1 }));
  }

  private setUserSession = async (userSession: UserSession) => {
    this.graphQLClient.setHeader('Authorization', `OAuth ${userSession.authToken}`);
    this.setState({ session: userSession });
    await this.loadProjects();
  }

  private handleMainMessage = (_event: any, message: any) => {
    switch (message.action) {
      case 'create project':
        this.showCreateProjectDialog();
        break;
      case 'open project':
        this.openProject();
        break;
      case 'check certificate':
        this.showCheckCertificateDialog(message.exceptionRequest);
        break;
      default:
        console.warn(`unexpected main message "${message.action}"`);
    }
  }

  private changePath = async (filePath: string) => {
    const { projectReferences, currentProjectIndex } = this.state;
    const projectReference = projectReferences[currentProjectIndex];
    projectReference.filePath = filePath;
    this.setState({ projectReferences });
    localStorage.projectReferences = projectReferences;
    await this.loadProject(projectReference);
  }

  private removeProject = async () => {
    const projectReferences = this.state.projectReferences.filter((_, index) => index !== this.state.currentProjectIndex);
    const currentProjectIndex = 0;
    this.setState({
      currentProject: undefined,
      currentProjectIndex,
      currentView: NavItem.ProjectOverview,
      projectReferences,
    });
    localStorage.currentProjectIndex = currentProjectIndex;
    localStorage.projectReferences = projectReferences;
    this.setState({ currentProject: undefined, isReady: false });
    if (projectReferences.length) {
      await this.loadProject(projectReferences[currentProjectIndex]);
    }
    this.setState({ isReady: true });
  }

  private hideReleaseNotes = () => {
    this.setState({ showingReleaseNotes: false });
  }

  private showReleaseNotes = () => {
    this.setState({ showingReleaseNotes: true });
  }

  private toggleRigNav = () => {
    this.setState((previousState) => ({ isRigNavOpen: !previousState.isRigNavOpen }));
  }

  private getClass(baseName: string, isRigNavOpen: boolean) {
    return classNames(baseName, {
      [`${baseName}--icon`]: !isRigNavOpen,
    });
  }

  private viewInBrowser = (viewId: string) => {
    const { currentProjectIndex } = this.state;
    viewInBrowser(localStorage.projectReferences[currentProjectIndex].filePath, this.currentProject.secret, viewId);
  }

  public render() {
    const { currentProjectIndex, currentView, isReady, isRigNavOpen, isUpToDate, openingProject, projectReferences, rigContext, themeClass } = this.state;
    let content: JSX.Element;

    if (!isReady) {
      content = <StyledLayout background={Background.Alt} className="rig__update">Loading...</StyledLayout>;
    } else if (!isUpToDate) {
      content = <StyledLayout background={Background.Alt} className="rig__update">Downloading update...</StyledLayout>;
    } else if (!this.state.session) {
      content = <WelcomePage setUserSession={this.setUserSession} />;
    } else {
      const currentProject = this.currentProject;
      const rigNavClassName = this.getClass('rig__rig-nav', isRigNavOpen);
      const rigViewClassName = this.getClass('rig__view', isRigNavOpen);
      content = <>
        <div className="rig__corner">
          <DropdownMenu
            action={
              <div className="rig__corner-action">Add Project</div>
            }
            menu={
              <>
                <Button ariaLabel="Rig:Create Project" overlay type={ButtonType.Text} onClick={this.showCreateProjectDialog}>Create Project</Button>
                <Button ariaLabel="Rig:Open Project" overlay type={ButtonType.Text} onClick={this.openProject}>Open Project</Button>
              </>
            }
            menuClassName="rig__corner-menu"
          />
        </div>
        <div className="rig__project-nav">
          <ProjectNav currentProjectIndex={currentProjectIndex} projectReferences={projectReferences} logout={this.logout} selectProject={this.selectProject} />
        </div>
        {this.state.showingLoadFailureView ? (
          <LoadFailureView projectFilePath={projectReferences[currentProjectIndex].filePath} projectName={projectReferences[currentProjectIndex].name}
            onChangePath={this.changePath} onRemove={this.removeProject} />
        ) : currentProject ? <>
          <div className={rigNavClassName}>
            <RigNav currentView={currentView} isOpen={isRigNavOpen} onToggle={this.toggleRigNav} selectView={this.selectView} showReleaseNotes={this.showReleaseNotes} />
          </div>
          <div className={rigViewClassName}>
            {currentView === NavItem.ProjectOverview && <ProjectView
              rigProject={currentProject}
              userId={this.userId}
              closeProject={this.closeProject}
              onChange={this.updateProject}
              refreshViews={this.refreshViews}
            />}
            {rigContext && <RigContext.Provider value={rigContext}>
              <ExtensionViewContainer
                key={`ExtensionViewContainer${this.state.extensionsViewContainerKey}`}
                authToken={this.session.authToken}
                isConsoleWide={!isRigNavOpen}
                isDisplayed={currentView === NavItem.ExtensionViews}
                rigProject={currentProject}
                onCreateExtensionView={this.createExtensionView}
                onDeleteExtensionView={this.deleteExtensionView}
                onUpdateExtensionViews={this.updateExtensionViews}
                onViewInBrowser={this.viewInBrowser}
              />
              {currentView === NavItem.ConfigurationService && <ConfigurationServiceView
                authToken={this.session.authToken}
                rigProject={currentProject}
              />}
            </RigContext.Provider>}
            {currentView === NavItem.ProductManagement &&
              <ProductsView clientId={currentProject.manifest.id} isBitsEnabled={currentProject.manifest.bitsEnabled} token={this.session.authToken} version={currentProject.manifest.version} />}
          </div>
        </> : <WelcomePage showCreateProjectDialog={this.showCreateProjectDialog} />}
        {openingProject && <OpenProjectDialog
          {...openingProject}
          userId={this.userId}
          closeHandler={this.cancelOpeningProject}
          openHandler={this.addProject}
        />}
        {this.state.exceptionRequest && <CheckCertificateDialog
          exceptionRequest={this.state.exceptionRequest}
          closeHandler={this.closeCheckCertificateDialog}
        />}
        {this.state.showingCreateProjectDialog && <CreateProjectWizard
          userId={this.userId}
          closeHandler={this.closeProjectDialog}
          saveHandler={this.addProject}
        />}
        {this.state.showingReleaseNotes && <ReleaseNotesDialog onClose={this.hideReleaseNotes} />}
      </>;
    }

    return (
      <div className={classNames('rig', 'tw-root--hover', themeClass)}>
        {content}
      </div>
    );
  }
}
