import { Product } from '../core/models/product';
import {
  addCertificateException,
  checkForUpdate,
  clearAuth,
  createProject,
  fetchAuth,
  fetchChannelConfigurationSegments,
  fetchExamples,
  fetchExtensionManifest,
  fetchExtensionSecret,
  fetchGlobalConfigurationSegment,
  fetchHostingStatus,
  fetchProducts,
  fetchProject,
  fetchUser,
  hasNode,
  hasVisualStudioCode,
  hostFrontend,
  launchVisualStudioCode,
  loadFile,
  openUrl,
  saveConfigurationSegment,
  saveFile,
  saveProduct,
  sendEvent,
  setHttpOption,
  setProject,
  showContextMenu,
  showMessageBox,
  showOpenDialog,
  startBackend,
  startFrontend,
  stopHosting,
  StopOptions,
  viewInBrowser,
} from './api';
import {
  mockFetchError,
  mockFetchForExtensionManifest,
  mockFetchForExtensionSecret,
  mockFetchForUserInfo,
  mockFetchProducts,
  mockSaveProduct,
  mockEmptyResponse,
  mockFetch400,
  mockFetch500,
} from '../tests/utils';
import { RigProject } from '../core/models/rig';

let globalAny = global as any;
const token = 'token';
const electron = (window as any).electron;

describe('api', () => {
  describe('API.fetch', () => {
    it('throws on 400', async function() {
      const { error: error_, warn: warn_ } = console;
      const [error, warn] = [jest.fn(), jest.fn()];
      globalAny.console = { error, warn };
      globalAny.fetch = jest.fn().mockImplementation(mockFetch400);
      try {
        await fetchUser(token);
      } catch (ex) {
        expect(ex.message).toBe('400 error');
      }
      expect(error).toHaveBeenCalled();
      expect(warn).toHaveBeenCalled();
      globalAny.console.error = error_;
      globalAny.console.warn = warn_;
    });

    it('throws on 500', async function() {
      const { error: error_, warn: warn_ } = console;
      const [error, warn] = [jest.fn(), jest.fn()];
      globalAny.console = { error, warn };
      globalAny.fetch = jest.fn().mockImplementation(mockFetch400);
      globalAny.fetch = jest.fn().mockImplementation(mockFetch500);
      try {
        await fetchUser(token);
      } catch (ex) {
        expect(ex.message).toBe('500 error');
      }
      expect(error).toHaveBeenCalled();
      expect(warn).toHaveBeenCalled();
      globalAny.console.error = error_;
      globalAny.console.warn = warn_;
    });

    it('throws an exception on failed fetch', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockFetchError);
      try {
        await fetchUser(token);
      } catch (ex) {
        expect(ex).toBe('Fake error');
      }
    });
  });

  it('addCertificateException', async function() {
    const certificateException = { isAllowed: true, isOrigin: true, value: 'test.com' };
    await addCertificateException(certificateException);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/certificate-exception/add', certificateException);
  });

  describe('checkForUpdate', () => {
    it('succeeds', async function() {
      await checkForUpdate();
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/update', undefined);
    });
  });

  describe('clearAuth', () => {
    it('succeeds', async function() {
      await clearAuth();
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/clear-auth', {});
    });
  });

  describe('createProject', () => {
    it('succeeds', async function() {
      const codeGenerationOption = 'codeGenerationOption';
      const exampleId = 0;
      const project = {} as RigProject;
      await createProject(codeGenerationOption, project, exampleId);
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/project', {
        codeGenerationOption,
        exampleId,
        project,
      });
    });
  });

  describe('fetchAuth', () => {
    it('succeeds', async function() {
      const expected = 'auth';
      electron.response = expected;
      const actual = await fetchAuth();
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/auth', undefined);
      expect(actual).toBe(expected);
    });
  });

  describe('fetchChannelConfigurationSegments', () => {
    it('succeeds', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockEmptyResponse);
      await fetchChannelConfigurationSegments('clientId', 'userId', 'channelId', 'secret');
      expect(globalAny.fetch).toHaveBeenCalledTimes(1);
    });

    it('returns broadcaster and developer', async function() {
      globalAny.fetch = jest.fn().mockImplementation(() => Promise.resolve({
        json: () => ({
          'broadcaster:channelId': {},
          'developer:channelId': {},
        })
      }));
      await fetchChannelConfigurationSegments('clientId', 'userId', 'channelId', 'secret');
      expect(globalAny.fetch).toHaveBeenCalledTimes(1);
    });
  });

  describe('fetchExamples', () => {
    it('succeeds', async function() {
      const expected = {};
      electron.response = expected;
      const actual = await fetchExamples();
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/examples', undefined);
      expect(actual).toBe(expected);
    });
  });

  describe('fetchExtensionManifest', () => {
    it('returns data', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockFetchForExtensionManifest);
      const data = await fetchExtensionManifest('clientId', 'version');
      expect(data).toBeDefined();
    });

    it('throws an exception on invalid response', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockEmptyResponse);
      expect.assertions(1);
      try {
        await fetchExtensionManifest('clientId', 'version');
      } catch (ex) {
        expect(ex.message).toBe('Unable to retrieve extension manifest; please verify your client ID, secret, and version');
      }
    });
  });

  describe('fetchExtensionSecret', () => {
    it('returns data', async function() {
      globalAny.fetch = jest.fn().mockImplementation(() => mockFetchForExtensionSecret(1));
      const extensionId = 'extensionId';
      const data = await fetchExtensionSecret(extensionId);
      expect(data).toBeDefined();
    });

    it('throws an exception on lack of secrets', async function() {
      globalAny.fetch = jest.fn().mockImplementation(() => mockFetchForExtensionSecret(0));
      expect.assertions(1);
      const extensionId = 'extensionId';
      try {
        await fetchExtensionSecret(extensionId);
      } catch (ex) {
        expect(ex.message).toBe(`No active secret for extension ${extensionId}`);
      }
    });
  });

  describe('fetchGlobalConfigurationSegment', () => {
    it('succeeds', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockEmptyResponse);
      await fetchGlobalConfigurationSegment('clientId', 'userId', 'secret');
      expect(globalAny.fetch).toHaveBeenCalledTimes(1);
    });
  });

  describe('fetchHostingStatus', () => {
    it('succeeds', async function() {
      await fetchHostingStatus();
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/status', undefined);
    });
  });

  describe('fetchUser', () => {
    beforeEach(() => {
      globalAny.fetch = jest.fn().mockImplementation(mockFetchForUserInfo);
    });

    it('returns data', async function() {
      const data = await fetchUser(token);
      expect(data).toBeDefined();
    });

    it('throws an exception on invalid response', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockEmptyResponse);
      expect.assertions(1);
      try {
        await fetchUser(token);
      } catch (ex) {
        expect(ex.message).toBe(`Invalid server response for access token ${token}`);
      }
    });

    it('returns null if user id not found', async function() {
      globalAny.fetch = jest.fn().mockImplementation(() => Promise.resolve({ json: () => [] }));
      const data = await fetchUser(token, '1');
      expect(data).toBe(null);
    });

    it('returns null if user name not found', async function() {
      globalAny.fetch = jest.fn().mockImplementation(() => Promise.resolve({ json: () => [] }));
      const data = await fetchUser(token, '-');
      expect(data).toBe(null);
    });
  });

  describe('fetchProducts', () => {
    it('serializes products correctly', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockFetchProducts);
      const products = await fetchProducts('clientId', '');
      expect(products).toHaveLength(2);
      products.forEach((product: Product) => {
        expect(product).toMatchObject({
          sku: expect.any(String),
          displayName: expect.any(String),
          amount: expect.any(Number),
          inDevelopment: expect.any(Boolean),
          broadcast: expect.any(Boolean),
        });
      });
    });

    it('throws an exception on invalid response', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockEmptyResponse);
      expect.assertions(1);
      try {
        await fetchProducts('clientId', token);
      } catch (ex) {
        expect(ex.message).toBe(`Invalid server response for access token ${token}`);
      }
    });
  });

  describe('fetchProject', () => {
    it('succeeds', async function() {
      globalAny.fetch = jest.fn().mockImplementation(mockFetchForExtensionManifest);
      const referenceId = 'referenceId';
      const externalSpecification = await fetchProject(referenceId);
      expect(externalSpecification).toBeDefined();
    });
  });

  it('hasNode', async () => {
    await hasNode();
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/has-node', {});
  });

  it('hasVisualStudioCode', async () => {
    await hasVisualStudioCode();
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/vs-code/check', {});
  });

  it('launchVisualStudioCode', async () => {
    const projectFolderPath = 'projectFolderPath';
    await launchVisualStudioCode(projectFolderPath);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/vs-code/launch', { projectFolderPath });
  });

  describe('loadFile', () => {
    it('succeeds', async function() {
      const loadFilePath = 'loadFilePath';
      await loadFile(loadFilePath);
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/load', { loadFilePath });
    });
  });

  it('openUrl', async () => {
    const url = 'url';
    await openUrl(url);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/open-url', { url });
  });

  describe('saveConfigurationSegment', () => {
    beforeEach(function() {
      globalAny.fetch = jest.fn().mockImplementation(mockSaveProduct);
    });

    it('succeeds with channel', async function() {
      await saveConfigurationSegment('clientId', 'userId', 'secret', 'segment', 'channelId', 'content', 'version');
    });

    it('succeeds with global', async function() {
      await saveConfigurationSegment('clientId', 'userId', 'secret', 'global', '', 'content', 'version');
    });
  });

  describe('saveFile', () => {
    it('succeeds', async function() {
      const saveFilePath = 'saveFilePath';
      const fileContent = 'fileContent';
      await saveFile(saveFilePath, fileContent);
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/save', { saveFilePath, fileContent });
    });
  });

  describe('saveProduct', () => {
    beforeEach(function() {
      globalAny.fetch = jest.fn().mockImplementation(mockSaveProduct);
    });

    it('succeeds', async function() {
      await saveProduct('clientId', token, {} as Product);
    });

    it('calls the API with the correct parameters', async function() {
      const product = {
        sku: 'sku',
        displayName: 'name',
        amount: 100,
        inDevelopment: true,
        broadcast: false,
        deprecated: false,
        dirty: true,
        savedInCatalog: false,
      };

      await saveProduct('clientId', token, product);
      expect(globalAny.fetch).toHaveBeenCalledWith('https://api.twitch.tv/v5/bits/extensions/twitch.ext.clientId/products/put', {
        method: 'POST',
        body: JSON.stringify({
          product: {
            domain: 'twitch.ext.clientId',
            sku: 'sku',
            displayName: 'name',
            cost: {
              amount: 100,
              type: 'bits',
            },
            inDevelopment: true,
            broadcast: false,
            expiration: null,
          }
        }),
        headers: {
          Accept: 'application/vnd.twitchtv.v5+json; charset=UTF-8',
          Authorization: `OAuth ${token}`,
          'Client-Id': 'clientId',
          'Content-Type': 'application/json; charset=UTF-8',
        },
      });
    })
  });

  it('sendEvent', async function() {
    const eventName = 'eventName';
    const eventData = {};
    await sendEvent(eventName, eventData);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/event', { eventName, eventData });
  });

  it('setHttpOption', async function() {
    await setHttpOption(true);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/allow-http', { wantsHttp: true });
  });

  it('setProject', async function() {
    const userId = 'userId';
    const project = {} as RigProject;
    await setProject(userId, project);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/set-project', { userId, project });
  });

  it('showContextMenu', async function() {
    await showContextMenu();
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/context', {});
  });

  it('showMessageBox', async function() {
    const expected = 'showMessageBox';
    electron.response = expected;
    const options = {};
    const actual = await showMessageBox(options);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/message-box', options);
    expect(actual).toBe(expected);
  });

  it('showOpenDialog', async function() {
    const expected = 'showOpenDialog';
    electron.response = expected;
    const options = {};
    const actual = await showOpenDialog(options);
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/dialog', options);
    expect(actual).toBe(expected);
  });

  it('hostFrontend', async function() {
    await hostFrontend('frontendFolderPath', 4349, 'projectFolderPath');
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/frontend', {
      frontendFolderPath: 'frontendFolderPath',
      port: 4349,
      projectFolderPath: 'projectFolderPath',
    });
  });

  it('startBackend', async () => {
    await startBackend('backendCommand', 'workingFolderPath');
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/backend', {
      backendCommand: 'backendCommand',
      workingFolderPath: 'workingFolderPath',
    });
  });

  it('startFrontend', async () => {
    await startFrontend('frontendCommand', 'frontendFolderPath', 'projectFolderPath');
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/frontend', {
      frontendCommand: 'frontendCommand',
      frontendFolderPath: 'frontendFolderPath',
      projectFolderPath: 'projectFolderPath',
    });
  });

  it('stopHosting', async function() {
    await stopHosting();
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/stop', {
      stopOptions: StopOptions.Backend | StopOptions.Frontend,
    });
  });

  it('viewInBrowser', async function() {
    const expected = { projectFilePath: 'projectFilePath', secret: 'secret', viewId: 'viewId' };
    await viewInBrowser('projectFilePath', 'secret', 'viewId');
    expect(electron.ipcRenderer.once).toHaveBeenCalled();
    expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/browser', expected);
  });

  it('fails local API', async function() {
    const errorMessage = 'error';
    electron.response = errorMessage;
    try {
      await fetchHostingStatus();
      throw new Error('unexpected');
    } catch (ex) {
      expect(electron.ipcRenderer.once).toHaveBeenCalled();
      expect(electron.ipcRenderer.send).toHaveBeenCalledWith('api', '/status', undefined);
      expect(ex.message).toBe(errorMessage);
    }
  });
});
