import { Product, DeserializedProduct } from '../core/models/product';
import { ExtensionManifest } from '../core/models/manifest';
import { toCamelCase } from '../util/case';
import { createConfigurationToken } from './token';
import electron from '../electron';
import { RigProject, RigExtensionView } from '../core/models/rig';
import env from '../env';
import { LocalStorage } from './local-storage';
import { CertificateException } from '../core/models/certificate-exception';
import { Segment } from 'extension-coordinator';

class Api {
  private isLocal: boolean;
  private url: string;

  constructor({ isLocal, url }: { isLocal: boolean, url?: string; }) {
    this.isLocal = isLocal;
    this.url = url || 'https://api.twitch.tv';
  }

  public async get<T>(path: string, headers?: HeadersInit): Promise<T> {
    return await this.fetch<T>('GET', path, headers);
  }

  public async post<T = void>(path: string, body: any, headers?: HeadersInit): Promise<T> {
    return await this.fetch<T>('POST', path, headers, body);
  }

  public async put<T = void>(path: string, body: any, headers?: HeadersInit): Promise<T> {
    return await this.fetch<T>('PUT', path, headers, body);
  }

  public async fetch<T>(method: 'GET' | 'POST' | 'PUT', path: string, headers?: HeadersInit, body?: any): Promise<T> {
    if (this.isLocal) {
      if (!electron) {
        return await externalApi.fetch<T>(method, path, headers, body);
      }
      const { ipcRenderer } = electron;
      const p = new Promise<T>((resolve, reject) => {
        ipcRenderer.once(`api${path}`, (_event: any, error: string, response: T) => {
          if (error) {
            reject(new Error(error));
          } else {
            resolve(response);
          }
        });
        ipcRenderer.send('api', path, body);
      });
      return await p;
    }
    const overridableHeaders: HeadersInit = {
      Accept: 'application/vnd.twitchtv.v5+json; charset=UTF-8',
    };
    if (body) {
      overridableHeaders['Content-Type'] = 'application/json; charset=UTF-8';
    }
    headers = headers ? Object.assign(overridableHeaders, headers) : overridableHeaders;
    const request: RequestInit = {
      method,
      headers,
    };
    if (headers.cookie === '') {
      request.credentials = 'omit';
    }
    if (body) {
      request.body = JSON.stringify(body);
    }
    const url = new URL(path, this.url);
    const response = await fetch(url.toString(), request);
    if (response.status >= 400) {
      console.warn(request);
      console.error(response);
      const message = 'Cannot access Twitch API.  Try again later.';
      throw new Error(await response.json().then((json) => json.message || message).catch(() => message));
    } else if (response.status !== 204) {
      return await response.json() as T;
    }
    return undefined as unknown as T;
  }
}

/* eslint no-restricted-globals: "off" */
const externalApi = new Api({ isLocal: false, url: location.href });
/* eslint no-restricted-globals: "error" */
const localApi = new Api({ isLocal: true });
const onlineApi = new Api({ isLocal: false });

export async function sendEvent(eventName: string, eventData: { [key: string]: boolean | string; }) {
  await localApi.post('/event', { eventName, eventData });
}

export async function checkForUpdate() {
  await localApi.get<void>('/update');
}

export async function createProject(codeGenerationOption: string, project: RigProject, exampleId?: number): Promise<string> {
  const path = '/project';
  return await localApi.post<string>(path, { codeGenerationOption, exampleId, project });
}

export async function hostFrontend(frontendFolderPath: string, port: number, projectFolderPath: string) {
  const path = '/frontend';
  await localApi.post(path, { frontendFolderPath, port, projectFolderPath });
}

export async function startFrontend(frontendCommand: string, frontendFolderPath: string, projectFolderPath: string) {
  const path = '/frontend';
  await localApi.post(path, { frontendCommand, frontendFolderPath, projectFolderPath });
}

export async function startBackend(backendCommand: string, workingFolderPath: string) {
  const path = '/backend';
  await localApi.post(path, { backendCommand, workingFolderPath });
}

export interface HostingStatus {
  isBackendRunning: boolean;
  isFrontendRunning: boolean;
  port?: number;
}

export async function fetchHostingStatus(): Promise<HostingStatus> {
  const path = '/status';
  return await localApi.get<HostingStatus>(path);
}

export interface StopResult {
  backendResult: string;
  frontendResult: string;
}

export enum StopOptions {
  Backend = 1,
  Frontend = 2,
}

export async function stopHosting(stopOptions?: StopOptions): Promise<StopResult> {
  const path = '/stop';
  return await localApi.post<StopResult>(path, { stopOptions: stopOptions || (StopOptions.Backend | StopOptions.Frontend) });
}

export interface Example {
  id: number;
  title: string;
  description: string;
  repository: string;
  frontendFolderName: string;
  frontendCommand: string;
  isGuide: boolean;
  backendFolderName?: string;
  backendCommand?: string;
  npm: string[];
  expectedDuration: number;
}

export async function fetchExamples(): Promise<Example[]> {
  const path = '/examples';
  return await localApi.get<Example[]>(path);
}

interface ExtensionSecret {
  content: string;
  active: string;
  expires: string;
}

interface ExtensionSecretGenerationResponse {
  form_version: number;
  secrets: ExtensionSecret[];
}

export async function fetchExtensionSecret(extensionId: string): Promise<string> {
  const { rigLogin } = new LocalStorage();
  if (!rigLogin) {
    throw new Error('No authentication token available');
  }

  const response = await onlineApi.get<ExtensionSecretGenerationResponse>(
    `/kraken/extensions/${extensionId}/auth/secret`,
    {
      'Content-Type': 'application/json',
      'Accept': 'application/vnd.twitchtv.v5+json; charset=UTF-8',
      'Authorization': `OAuth ${rigLogin.authToken}`,
      'Client-Id': env.RIG_CLIENT_ID,
    }
  );
  const secret = response.secrets.length > 1 ?
    response.secrets.find((s) => Date.now() >= Date.parse(s.active)) :
    response.secrets[0];
  if (secret) {
    return secret.content;
  } else {
    throw new Error(`No active secret for extension ${extensionId}`);
  }
}

export async function fetchExtensionManifest(id: string, version: string): Promise<ExtensionManifest> {
  const { rigLogin } = new LocalStorage();
  if (!rigLogin) {
    throw new Error('No authentication token available');
  }
  const response = await onlineApi.get<{ views: any; }>(`/extensions/${id}/${version}`, {
    // This calls Visage which calls Cartman for a capatibility token.  Visage
    // supports both "OAuth ..." and "Bearer ..." but Cartman supports only "OAuth ...".
    Authorization: `OAuth ${rigLogin.authToken}`,
    'Client-Id': id,
    'cookie': '',
  });
  if (response) {
    if (response.views.component) {
      response.views.component.autoScale = response.views.component.autoscale;
      delete response.views.component.autoscale;
    }
    const manifest = toCamelCase(response) as ExtensionManifest;
    return manifest;
  }
  throw new Error('Unable to retrieve extension manifest; please verify your client ID, secret, and version');
}

interface UsersResponse {
  data: {
    broadcaster_type: string;
    description: string;
    display_name: string;
    email: string;
    id: string;
    login: string;
    offline_image_url: string;
    profile_image_url: string;
    type: string;
    view_count: number;
  }[];
}

export async function fetchUser(token: string, idOrLogin?: string, isId?: boolean) {
  const path = `/helix/users${idOrLogin ? `?${isId ? 'id' : 'login'}=${idOrLogin}` : ''}`;
  const response = await onlineApi.get<UsersResponse>(path, {
    Authorization: `Bearer ${token}`,
    'Client-Id': env.RIG_CLIENT_ID,
  });
  const { data } = response;
  if (data && data.length) {
    return data[0];
  }
  if (idOrLogin) {
    // Did not find that user.
    return null;
  }
  throw new Error(`Invalid server response for access token ${token}`);
}

interface ProductsResponse {
  products: DeserializedProduct[];
}

export async function fetchProducts(clientId: string, token: string): Promise<Product[]> {
  const path = `/v5/bits/extensions/twitch.ext.${clientId}/products?includeAll=true`;
  const response = await onlineApi.get<ProductsResponse>(path, {
    Authorization: `OAuth ${token}`,
    'Client-Id': clientId,
  });
  if (response.products) {
    return response.products.map((p: DeserializedProduct) => ({
      sku: p.sku || '',
      displayName: p.displayName || '',
      amount: p.cost ? Number(p.cost.amount) : 1,
      inDevelopment: p.inDevelopment || false,
      broadcast: p.broadcast || false,
      deprecated: p.expiration ? Date.parse(p.expiration) <= Date.now() : false,
      dirty: false,
      savedInCatalog: true,
    } as Product));
  }
  throw new Error(`Invalid server response for access token ${token}`);
}

export async function saveProduct(clientId: string, token: string, product: Product) {
  const path = `/v5/bits/extensions/twitch.ext.${clientId}/products/put`;
  const deserializedProduct = {
    domain: 'twitch.ext.' + clientId,
    sku: product.sku,
    displayName: product.displayName,
    cost: {
      amount: product.amount,
      type: 'bits',
    },
    inDevelopment: product.inDevelopment,
    broadcast: product.broadcast,
    expiration: product.deprecated ? new Date(Date.now()).toISOString() : null,
  };
  await onlineApi.post(path, { product: deserializedProduct }, {
    Authorization: `OAuth ${token}`,
    'Client-Id': clientId,
  });
}

interface SegmentRecordMap {
  [key: string]: {
    record: Segment;
  };
}

export async function fetchGlobalConfigurationSegment(clientId: string, userId: string, secret: string): Promise<Segment | undefined> {
  const path = `/extensions/${clientId}/configurations/segments/global`;
  const headers = {
    Authorization: `Bearer ${createConfigurationToken(secret, userId)}`,
    'Client-Id': clientId,
  };
  const segmentMap = await onlineApi.get<SegmentRecordMap>(path, headers);
  const global = segmentMap['global:'];
  return global ? global.record : undefined;
}

export interface SegmentMap {
  broadcaster?: Segment;
  developer?: Segment;
}

export async function fetchChannelConfigurationSegments(clientId: string, userId: string, channelId: string, secret: string): Promise<SegmentMap> {
  const path = `/extensions/${clientId}/configurations/channels/${channelId}`;
  const headers = {
    Authorization: `Bearer ${createConfigurationToken(secret, userId)}`,
    'Client-Id': clientId,
  };
  const segmentMap = await onlineApi.get<SegmentRecordMap>(path, headers);
  const broadcaster = segmentMap[`broadcaster:${channelId}`];
  const developer = segmentMap[`developer:${channelId}`];
  const segments = {} as SegmentMap;
  if (broadcaster) {
    segments.broadcaster = broadcaster.record;
  }
  if (developer) {
    segments.developer = developer.record;
  }
  return segments;
}

export async function saveConfigurationSegment(clientId: string, userId: string, secret: string, segment: string, channelId: string, content: string, version: string) {
  const path = `/extensions/${clientId}/configurations`;
  const headers = {
    Authorization: `Bearer ${createConfigurationToken(secret, userId)}`,
    'Client-Id': clientId,
  };
  const body: { [key: string]: string; } = {
    segment,
    version,
    content,
  };
  if (segment !== 'global') {
    body.channel_id = channelId;
  }
  await onlineApi.put(path, body, headers);
}

export async function loadFile(loadFilePath: string): Promise<string> {
  const path = '/load';
  return await localApi.post<string>(path, { loadFilePath });
}

export async function saveFile(saveFilePath: string, fileContent: string) {
  const path = '/save';
  await localApi.post(path, { saveFilePath, fileContent });
}

export async function showContextMenu() {
  const path = '/context';
  await localApi.post(path, {});
}

export async function showMessageBox(options: any): Promise<number> {
  const path = '/message-box';
  return await localApi.post<number>(path, options);
}

export async function showOpenDialog(options: any): Promise<string | undefined> {
  const path = '/dialog';
  return await localApi.post<string | undefined>(path, options);
}

export async function clearAuth() {
  const path = '/clear-auth';
  await localApi.post(path, {});
}

export async function fetchAuth(): Promise<string> {
  const path = '/auth';
  return await localApi.get<string>(path);
}

export async function viewInBrowser(projectFilePath: string, secret: string, viewId: string) {
  const path = '/browser';
  await localApi.post(path, { projectFilePath, secret, viewId });
}

export async function setHttpOption(wantsHttp: boolean) {
  const path = '/allow-http';
  await localApi.post(path, { wantsHttp });
}

export async function hasVisualStudioCode(): Promise<boolean> {
  const path = '/vs-code/check';
  return await localApi.post<boolean>(path, {});
}

export async function launchVisualStudioCode(projectFolderPath: string) {
  const path = '/vs-code/launch';
  await localApi.post(path, { projectFolderPath });
}

export async function addCertificateException(certificateException: CertificateException) {
  const path = '/certificate-exception/add';
  await localApi.post(path, certificateException);
}

export async function setProject(userId: string, project: RigProject) {
  const path = '/set-project';
  await localApi.post(path, { userId, project });
}

interface ExternalSpecification {
  project: RigProject;
  secret: string;
  view: RigExtensionView;
}

export async function fetchProject(referenceId: string): Promise<ExternalSpecification> {
  return await externalApi.get<ExternalSpecification>(`/project/${referenceId}`);
}

export async function openUrl(url: string) {
  const path = '/open-url';
  await localApi.post(path, { url });
}

export async function hasNode(): Promise<boolean> {
  const path = '/has-node';
  return await localApi.post<boolean>(path, {});
}
