import { createExtensionObject, createExtensionToken } from './extension';
import { ExtensionManifest } from '../core/models/manifest';
import { ExtensionTokenPayload, Role } from './token';
import { verify } from 'jsonwebtoken';
import { ViewerType } from '../core/models/rig';
import { createExtensionManifestForTest, createExtensionObjectForTest } from '../tests/constants/extension';

type TestTokenPayload = ExtensionTokenPayload & {
  exp: number;
  iat: number;
};

describe('extension', () => {
  const manifest: ExtensionManifest = createExtensionManifestForTest();
  const index = '0';
  const viewerType = ViewerType.LoggedOut;
  const channelId = 'test';
  const secret = 'test';
  const opaqueId = 'testOpaqueId';

  it('creates an extension with the correct data', () => {
    const expected = createExtensionObjectForTest();
    const result = createExtensionObject(manifest, index, viewerType, '', channelId, secret, opaqueId);
    const rx = /(-|\w)+\.(-|\w)+\.(-|\w)+/;
    result.token = result.token.replace(rx, 'token');
    expect(result).toEqual(expected);
  });
});

describe('createExtensionToken', () => {
  const secret = 'secret';
  const ouid = 'rig_ouid';
  const channelId = '999999999';
  const linkedUserId = '888888888';

  const LOGGED_OUT_PAYLOAD = {
    channel_id: channelId,
    opaque_user_id: 'ARIG' + ouid,
    pubsub_perms: {
      listen: ['broadcast'],
    },
    role: Role.Viewer,
  };

  const LOGGED_IN_UNLINKED_PAYLOAD = {
    channel_id: channelId,
    opaque_user_id: 'URIG' + ouid,
    pubsub_perms: {
      listen: ['broadcast'],
    },
    role: Role.Viewer,
  };

  const LOGGED_IN_LINKED_PAYLOAD = {
    channel_id: channelId,
    opaque_user_id: 'URIG' + ouid,
    pubsub_perms: {
      listen: ['broadcast'],
    },
    role: Role.Viewer,
    user_id: linkedUserId,
  };

  const BROADCASTER_PAYLOAD = {
    channel_id: channelId,
    opaque_user_id: 'U' + channelId,
    pubsub_perms: {
      listen: ['broadcast'],
      send: ['broadcast'],
    },
    role: Role.Broadcaster,
    user_id: channelId,
  };

  function expectPayload(payload: TestTokenPayload, opaqueUserId: string, pubsubPerms: {}, role: Role) {
    expect(payload.exp).toBeGreaterThan(Date.now() / 1000);
    expect(payload.opaque_user_id).toBe(opaqueUserId);
    expect(payload.pubsub_perms).toEqual(pubsubPerms);
    expect(payload.role).toBe(role);
    expect(payload.iat).toBeLessThan(payload.exp);
  }

  it('should create a token for logged-out users', () => {
    const token = createExtensionToken(ViewerType.LoggedOut, '', channelId, secret, ouid);
    const payload = verify(token, Buffer.from(secret, 'base64')) as TestTokenPayload;
    expectPayload(payload, LOGGED_OUT_PAYLOAD.opaque_user_id, { listen: ['broadcast', 'global'] }, LOGGED_OUT_PAYLOAD.role);
    expect(payload.user_id).toBeUndefined();
  });

  it('should create a token for unlinked logged-in users', () => {
    const token = createExtensionToken(ViewerType.LoggedIn, '', channelId, secret, ouid);
    const payload = verify(token, Buffer.from(secret, 'base64')) as TestTokenPayload;
    expectPayload(payload, LOGGED_IN_UNLINKED_PAYLOAD.opaque_user_id, { listen: ['broadcast', 'global'] }, LOGGED_IN_UNLINKED_PAYLOAD.role);
    expect(payload.user_id).toBeUndefined();
  });

  it('should create a token for linked logged-in users', () => {
    const token = createExtensionToken(ViewerType.LoggedIn, linkedUserId, channelId, secret, ouid);
    const payload = verify(token, Buffer.from(secret, 'base64')) as TestTokenPayload;
    expectPayload(payload, LOGGED_IN_LINKED_PAYLOAD.opaque_user_id, { listen: ['broadcast', 'global'] }, LOGGED_IN_LINKED_PAYLOAD.role);
    expect(payload.user_id).toBe(LOGGED_IN_LINKED_PAYLOAD.user_id);
  });

  it('should create a token for broadcaster users', () => {
    const token = createExtensionToken(ViewerType.Broadcaster, '', channelId, secret, ouid);
    const payload = verify(token, Buffer.from(secret, 'base64')) as TestTokenPayload;
    expectPayload(payload, BROADCASTER_PAYLOAD.opaque_user_id, { listen: ['broadcast', 'global'], send: ['broadcast'] }, BROADCASTER_PAYLOAD.role);
    expect(payload.user_id).toBe(BROADCASTER_PAYLOAD.user_id);
  });

  it('should create a token for the Rig', () => {
    const token = createExtensionToken(Role.Rig, '', channelId, secret, ouid);
    const payload = verify(token, Buffer.from(secret, 'base64')) as TestTokenPayload;
    expectPayload(payload, `ARIG${ouid}`, { listen: ['*'], send: ['*'] }, Role.Rig);
    expect(payload.user_id).toBeUndefined();
  });
});
