import * as React from 'react';
import './component.sass';
import env from '../../../env';
import { fetchAuth, fetchUser, showMessageBox } from '../../../util/api';
import electron from '../../../electron';
import { LocalStorage } from '../../../util/local-storage';
import { UserSession } from '../../../core/models/user-session';
import { parse } from 'querystring';

type NewWindowEvent = Event & { url: string };

// https://stackoverflow.com/a/47171216
declare global {
  interface HTMLElementEventMap {
    'new-window': NewWindowEvent;
  }
}

interface Props {
  setUserSession: (userSession: UserSession) => void;
}

export class SignInDialog extends React.Component<Props>{
  async componentDidMount() {
    try {
      const hash = parse(await fetchAuth());
      const accessToken = hash['access_token'];
      if (!accessToken) {
        throw new Error('no access token');
      }
      if (typeof accessToken !== 'string') {
        throw new Error('invalid access token');
      }
      const response = (await fetchUser(accessToken))!;
      const localStorage = new LocalStorage();
      const userSession: UserSession = {
        authToken: accessToken,
        displayName: response.display_name,
        email: response.email,
        expiration: Date.now() + 1000 * 60 * 60 * 24 * 55,
        id: response.id,
        login: response.login,
        profileImageUrl: response.profile_image_url || 'https://static-cdn.jtvnw.net/jtv_user_pictures/xarth/404_user_70x70.png',
      };
      localStorage.rigLogin = userSession;
      this.props.setUserSession(userSession);
    } catch (ex) {
      showMessageBox({ title: 'Sign-in Error', message: ex.message, type: 'error' });
    }
  }

  public render() {
    const { RIG_CLIENT_ID, RIG_REDIRECT_URL } = env;
    const rigAuthUrl = `https://id.twitch.tv/oauth2/authorize?client_id=${RIG_CLIENT_ID}&redirect_uri=${RIG_REDIRECT_URL}&response_type=token&scope=user:edit+user:read:email+extensions:edit:catalog`;
    // It isn't possible to use an iframe.  See https://stackoverflow.com/a/31812209.
    return (
      <div className="sign-in-dialog">
        <webview className="sign-in-dialog__frame" src={rigAuthUrl} ref={this.setWebview} />
      </div>
    );
  }

  private setWebview = (webview: HTMLWebViewElement | null) => {
    if (webview) {
      webview.addEventListener('new-window', (event: HTMLElementEventMap['new-window']) => {
        const url = new URL(event.url);
        if (url.protocol === 'http:' || url.protocol === 'https:') {
          electron.openExternal(event.url);
        }
      });
    }
  }
}
