/**
 * Gatsby Node
 *
 * This file sources all of the content, component information, and creates
 * the pages for the documentation website. It creates a GraphQL layer that
 * can be queried in the documentation components (in `/docs`).
 *
 * Resources:
 *  - https://www.gatsbyjs.org/docs/node-apis/
 */

require("dotenv").config();
const path = require("path");
const slash = require("slash");
const { last } = require("lodash");
const ForkTsCheckerWebpackPlugin = require("fork-ts-checker-webpack-plugin");
const siteMetadata = require("./gatsby-config").siteMetadata;
const gatsbyHelpers = require("./utils/gatsby-helpers");

/**
 * Gatsby looks through the data sources and creates nodes for each item. This
 * is when custom fields are added to the nodes. These fields include URL
 * segments for idenitfying files and building paths, formatting the page
 * title based on its filename, and getting the component properties and
 * comments.
 */
exports.onCreateNode = ({ node, actions }) => {
  if (node.internal.type !== "File" || !node.relativePath.match(/\.md$/)) {
    return;
  }

  const { createNodeField } = actions;
  const segments = gatsbyHelpers.getSegementsFromPath(node.relativePath);
  let tab = segments.pop();
  let path = `/${segments.map(s => s.slug).join("/")}`;
  let tabPath = `${path}${tab.slug !== "index" ? "/" + tab.slug : ""}`;
  let title = last(segments).title;
  let slug = last(segments).slug;

  // Create the fields on the node.
  if (segments) createNodeField({ node, name: `segments`, value: segments });
  if (title) createNodeField({ node, name: `title`, value: title });
  if (slug) createNodeField({ node, name: `slug`, value: slug });
  if (path) createNodeField({ node, name: `path`, value: path });
  if (tab) createNodeField({ node, name: `tab`, value: tab });
  if (tabPath) createNodeField({ node, name: `tabPath`, value: tabPath });
};

/**
 * Create the pages.
 */
exports.createPages = ({ graphql, actions }) => {
  const { createPage, createRedirect } = actions;

  return new Promise((resolve, reject) => {
    // Create an object of available templates.
    const templateNames = ["default-page"];

    const reducer = (accumulator, templateName) =>
      Object.assign(accumulator, {
        [templateName]: path.resolve(`src/templates/${templateName}.tsx`),
      });

    const templates = templateNames.reduce(reducer, {});

    // Query all markdown files in the `data` folder to create our pages.
    graphql(
      `
        {
          dataPages: allFile(
            sort: { fields: [relativePath], order: ASC }
            filter: {
              absolutePath: { regex: "/data/" }
              extension: { regex: "/md/" }
            }
          ) {
            edges {
              node {
                relativePath
                childMarkdownRemark {
                  frontmatter {
                    redirect
                  }
                }
                fields {
                  path
                  tab {
                    slug
                  }
                }
              }
            }
          }
        }
      `,
    ).then(result => {
      if (result.errors) {
        return reject(result.errors);
      }

      // Create all of the pages.
      result.data.dataPages.edges
        .map(p => p.node)
        .forEach(page => {
          const indexPagePath = page.fields.path;

          if (page.childMarkdownRemark.frontmatter.redirect) {
            // Create redirects paths with and without trailing slashes.
            ["", "/"].forEach(slash => {
              createRedirect({
                fromPath: indexPagePath + slash,
                isPermanent: true,
                redirectInBrowser: true,
                toPath: `${indexPagePath}/${page.childMarkdownRemark.frontmatter.redirect}`,
              });
            });

            // Need to create a page to prevent 404 errors because we can't
            // handle server configs on GitHub pages.
            createPage({
              path: indexPagePath,
              component: slash(templates["default-page"]),
              context: {
                pagePath: indexPagePath,
              },
            });

            return;
          }

          // So we don't have do duplicate settings between `index.md` and tab
          // files within a folder, we check the `index.md` file for settings
          // in the frontmatter.
          graphql(`
            {
              settingsFile: file (
                relativePath: { eq: "${page.relativePath.replace(
                  /(?!.*\/).+\.md$/,
                  "index.md",
                )}" }
              ) {
                childMarkdownRemark {
                  frontmatter {
                    title
                  }
                }
                fields {
                  title
                }
              }
            }
        `).then(result => {
            let pageTitle = gatsbyHelpers.getDisplayTitle(
              result.data.settingsFile,
            );

            // Don't append anything for the main tab title
            if (page.fields.tab.slug && page.fields.tab.slug !== "index") {
              pageTitle = `${pageTitle} | ${gatsbyHelpers.tabSlugToTitle(
                page.fields.tab.slug,
              )}`;
            }

            // Create the page.
            createPage({
              path: gatsbyHelpers.getPagePath(
                indexPagePath,
                page.fields.tab.slug,
              ),
              component: slash(templates["default-page"]),
              context: {
                pagePath: indexPagePath,
                tab: page.fields.tab.slug,
                title: pageTitle,
                showSidebarMenu: true,
              },
            });

            return;
          });
        });

      resolve();
    });
  });
};

/**
 * Modify the Webpack config.
 */
exports.onCreateWebpackConfig = ({ plugins, actions }) => {
  actions.setWebpackConfig({
    resolve: {
      modules: [path.resolve(__dirname), "node_modules"],
    },
    plugins: [
      new ForkTsCheckerWebpackPlugin({
        tslint: path.join(__dirname, "tslint.json"),
      }),
      plugins.define({
        "process.env": {
          NODE_ENV: JSON.stringify(
            process.env.NODE_ENV ? process.env.NODE_ENV : "development",
          ),
          SITE_METADATA: JSON.stringify(siteMetadata),
        },
      }),
    ],
  });
};
