import * as iam from "@aws-cdk/aws-iam";
import * as lambda from "@aws-cdk/aws-lambda";
import * as lambdaNodeJs from "@aws-cdk/aws-lambda-nodejs";
import * as cdk from "@aws-cdk/core";
import { Stack } from "@aws-cdk/core";
import { WEBSITE_NAME } from "../bin/consts";
import { CommonProps } from "./commonProps";

export interface AuthLambdaEdgeProps extends CommonProps {}

export class AuthLambdaEdgeStack extends Stack {
  public readonly handlerVersion: lambda.IVersion;

  constructor(scope: cdk.Construct, id: string, props: AuthLambdaEdgeProps) {
    super(scope, id, props);

    const name = `${props.prefix}-${WEBSITE_NAME}`;

    // Lambda@Edge requires both lambda, and edgelambda principal
    const lambdaTrustRole = new iam.Role(this, `${name}-lambda@edge-policy`, {
      assumedBy: new iam.CompositePrincipal(
        new iam.ServicePrincipal("lambda.amazonaws.com"),
        new iam.ServicePrincipal("edgelambda.amazonaws.com"),
      ),
      // Allows to log to cloudwatch
      managedPolicies: [
        iam.ManagedPolicy.fromManagedPolicyArn(
          this,
          id,
          "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole",
        ),
      ],
    });

    // Create lambda@edge handler for cloudfront distribution
    const originRequestHandler = new lambdaNodeJs.NodejsFunction(
      this,
      `${name}-OriginRequestHandler`,
      {
        entry: `${__dirname}/../../src/lambda/OriginRequestHandler.js`,
        handler: "originRequestHandler",
        runtime: lambda.Runtime.NODEJS_12_X,
        role: lambdaTrustRole,
      },
    );

    // Lambda@edge needs a numbered version to add to cloudfront
    this.handlerVersion = new lambda.Version(
      this,
      `${name}-OriginRequestHandler-version`,
      {
        lambda: originRequestHandler,
      },
    );
  }
}
