import * as React from "react";
import Moment from "react-moment";
import {
  AlignItems,
  Avatar,
  Background,
  BorderRadius,
  CoreText,
  Display,
  FlexWrap,
  FontSize,
  Layout,
  StyledLayout,
} from "twitch-core-ui";
import { GithubCommits } from "../../graphql-types";
import "./styles.scss";

interface Props {
  /** A GitHub file returned by a Gatsby GQL query.  */
  file: GithubCommits;
}

/**
 * Renders a list of commit items.
 */
export const Changelog = (props: Props) => {
  let commits: JSX.Element[] = [];

  if (props.file.items) {
    props.file.items.map(commit => {
      let commitAuthor: JSX.Element | undefined;
      let commitMeta: JSX.Element | undefined;

      if (commit.author) {
        commitAuthor = (
          <a href={commit.author.htmlUrl ? commit.author.htmlUrl : ""}>
            <Avatar
              alt={commit.author.login ? commit.author.login : ""}
              src={commit.author.avatarUrl}
              size={50}
              borderRadius={BorderRadius.Small}
              userLogin={null}
            />
          </a>
        );
      }

      if (commit.commit && commit.commit.committer) {
        commitMeta = (
          <div>
            <Moment format="MMMM D, YYYY">
              {commit.commit.committer.date || ""}
            </Moment>{" "}
            •{" "}
            {commit.author && commit.author.login ? (
              <a href={commit.author.htmlUrl ? commit.author.htmlUrl : ""}>
                {commit.author.login}
              </a>
            ) : (
              ""
            )}
            <CoreText fontSize={FontSize.Size5} bold>
              {commit.commit.message} (
              <a href={commit.htmlUrl ? commit.htmlUrl : ""}>
                {commit.sha ? commit.sha.substring(0, 7) : ""}
              </a>
              )
            </CoreText>
          </div>
        );
      }

      if (commit.sha && commit.sha !== "network") {
        commits.push(
          <Layout margin={{ y: 2 }} key={commit.sha}>
            <Layout
              display={Display.Flex}
              alignItems={AlignItems.Center}
              flexWrap={FlexWrap.NoWrap}
            >
              <Layout flexShrink={0}>{commitAuthor}</Layout>
              <Layout flexGrow={1} padding={{ x: 1 }}>
                {commitMeta}
              </Layout>
            </Layout>
          </Layout>,
        );
      } else {
        commits.push(
          <StyledLayout
            padding={{ x: 3, y: 2 }}
            background={Background.Alt2}
            borderRadius={BorderRadius.Medium}
            border
          >
            <CoreText>
              Unable to fetch GitHub data. Ensure that you are on a network that
              can access the Twitch Enterprise GitHub and the Github environment
              variables are set.
            </CoreText>
          </StyledLayout>,
        );
      }
    });
  }

  return <div>{commits}</div>;
};
