import * as Prism from "prismjs";
import * as React from "react";
import { renderToStaticMarkup } from "react-dom/server";
import * as renderHTML from "react-render-html";
import {
  AlignItems,
  BorderRadius,
  Color,
  Display,
  FlexDirection,
  FontSize,
  Interactable,
  InteractableType,
  JustifyContent,
  Layout,
  Position,
  StyledLayout,
  SVG,
  SVGAsset,
} from "twitch-core-ui";

import "./styles.scss";

// Add syntax highlighting support for additional languages.
import "prismjs/components/prism-go";
import "prismjs/components/prism-javascript";
import "prismjs/components/prism-json";

interface Props {
  children?: any;
  renderCode?: string;
  language?: string;
  truncate?: boolean;
}

interface State {
  collapsed?: boolean;
  canBeCollapsed?: boolean;
}

// If the code is taller than this many pixels, it should be hidden.
const DEFAULT_MAX_HEIGHT = 500;
// This is so we avoid a UX where the expand button expands us only a few more pixels.
const MAX_HEIGHT_THRESHOLD = 100;

/**
 * Any code snippet should be implemented using this component.
 */
export class CodeEditor extends React.Component<Props, State> {
  private codeEditorContent: HTMLDivElement | null;

  constructor(props: Props) {
    super(props);

    this.state = {
      collapsed: false,
      canBeCollapsed: props.truncate || false,
    };
  }

  public componentDidMount() {
    if (this.props.truncate && this.codeEditorContent !== null) {
      const heightTooTall =
        this.codeEditorContent.offsetHeight >=
        DEFAULT_MAX_HEIGHT + MAX_HEIGHT_THRESHOLD;
      this.setState({
        canBeCollapsed: heightTooTall,
        collapsed: heightTooTall,
      });
    }
  }

  public render() {
    let renderCode: string = "";
    let lineCount: number = 0;

    if (typeof window !== "undefined") {
      let textContent = window.document
        .createRange()
        .createContextualFragment(renderToStaticMarkup(this.props.children))
        .textContent;
      if (textContent) {
        renderCode = textContent.trim();
        lineCount = renderCode.split("\n").length;
      }
    }

    return (
      <StyledLayout
        className="code-editor"
        display={Display.Flex}
        flexDirection={FlexDirection.Column}
        color={Color.Overlay}
        borderRadius={BorderRadius.Medium}
      >
        <StyledLayout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          position={Position.Relative}
          flexGrow={1}
          elevation={2}
          borderRadius={BorderRadius.Small}
          refDelegate={ref => (this.codeEditorContent = ref)}
        >
          <pre
            style={{
              maxHeight:
                this.state.canBeCollapsed && this.state.collapsed
                  ? `${DEFAULT_MAX_HEIGHT}px`
                  : undefined,
            }}
          >
            <code>
              {this.highlightCode(this.props.children, this.props.language)}
            </code>
          </pre>

          {this.state.canBeCollapsed && this.state.collapsed && (
            <Layout
              className="code-editor__feather"
              padding={4}
              position={Position.Absolute}
              fullWidth
              attachBottom
              attachLeft
            />
          )}
        </StyledLayout>
        <Layout display={Display.HideAccessible}>
          <textarea value={renderCode} readOnly />
        </Layout>
        {this.state.canBeCollapsed && this.renderExpandToggle(lineCount)}
      </StyledLayout>
    );
  }

  private highlightCode = (code: string, language?: string) => {
    let prismLanguage = language && Prism.languages[language];
    let renderCode = code;
    if (prismLanguage) {
      renderCode = Prism.highlight(code, prismLanguage, language);
    }

    return renderHTML(renderCode);
  };

  private toggleCodeHeight = () => {
    this.setState({
      collapsed: !this.state.collapsed,
    });
  };

  private renderExpandToggle = (lineCount: number) => (
    <StyledLayout
      className="code-editor__toggle"
      fontSize={FontSize.Size7}
      borderRadius={{
        bottomLeft: BorderRadius.Medium,
        bottomRight: BorderRadius.Medium,
      }}
    >
      <Interactable
        type={InteractableType.Overlay}
        onClick={this.toggleCodeHeight}
      >
        <Layout
          display={Display.Flex}
          alignItems={AlignItems.Center}
          justifyContent={JustifyContent.Center}
          padding={{ x: 1, y: 0.5 }}
        >
          <SVG
            asset={this.state.collapsed ? SVGAsset.AngleDown : SVGAsset.AngleUp}
            width={16}
          />
          <Layout padding={{ left: 0.5 }}>
            {this.state.collapsed
              ? `Show All (${lineCount} lines)`
              : "Show Less"}
          </Layout>
        </Layout>
      </Interactable>
    </StyledLayout>
  );
}
