import { Location, LocationContext } from "@reach/router";
import * as classnames from "classnames";
import * as React from "react";
import { Display, Layout, Position } from "twitch-core-ui";
import { Header } from "../header";
import { SidebarMenu } from "../sidebar-menu";

import "twitch-core-ui/css/index-light-fallback.css";
import "./styles.scss";

interface Props {
  children: any;
}

type LayoutProps = Props & LocationContext;

interface PageProps {
  children: any;
  pageContext: PageContext;
}

interface PageContext {
  showSidebarMenu?: boolean;
}

/**
 * This component wraps every page on the site and contains high-order
 * components such as the Sidebar and SidebarMenu. As Gatsby v2, it uses
 * `StaticQuery` to perform the GraphQL query and pass data (including
 * Location information from @reach/router) to the page templates.
 */
export const SiteLayout = (props: PageProps) => {
  const children = props.children;
  const showSidebarMenu = props.pageContext.showSidebarMenu;

  return (
    <Location>
      {({ location, navigate }) =>
        renderLayout({ location, navigate, children }, showSidebarMenu)
      }
    </Location>
  );
};

const renderLayout = (props: LayoutProps, showSidebarMenu?: boolean) => {
  const classes: ClassValue = {
    layout: true,
    "layout--has-sidebar-menu": showSidebarMenu,
  };

  return (
    <>
      <Header location={props.location} navigate={props.navigate} />
      <Layout fullHeight className={classnames(classes)}>
        <Layout
          className="layout__sidebar"
          display={Display.Flex}
          position={Position.Fixed}
          attachTop
          attachLeft
        >
          {showSidebarMenu && (
            <SidebarMenu location={props.location} navigate={props.navigate} />
          )}
        </Layout>
        <Layout position={Position.Relative} fullHeight>
          {props.children}
        </Layout>
      </Layout>
    </>
  );
};
