import * as classnames from "classnames";
import * as React from "react";
import * as renderHTML from "react-render-html";
import {
  Aspect,
  AspectRatio,
  Background,
  Layout,
  Margin,
  StyledLayout,
  TextType,
  Title,
  TitleSize,
} from "twitch-core-ui";
import { DOMParser, XMLSerializer } from "xmldom";
import * as YAML from "yaml-js";
import { CodeEditor } from "../code-editor";
import { CodePreview } from "../code-preview";
import { BlockColumns } from "./components/block-columns";
import { BlockGuidelines } from "./components/block-guidelines";
import { BlockLinks } from "./components/block-links";
import { BlockMedia } from "./components/block-media";
import "./styles.scss";

export enum BaseSize {
  Default = 1,
  Smaller,
  Larger,
}

const BASE_SIZE_CLASSES = {
  [BaseSize.Default]: "markdown--size-default",
  [BaseSize.Smaller]: "markdown--size-smaller",
  [BaseSize.Larger]: "markdown--size-larger",
};

export interface Props {
  source: string;
  size?: BaseSize;
  hideCopyCodeButton?: boolean;
}

/**
 * All markdown should be implemented using this component.
 */
export class Markdown extends React.Component<Props, {}> {
  public render() {
    if (!this.props.source) {
      return <div />;
    }

    const classes: ClassValue = {};

    if (this.props.size) {
      classes[BASE_SIZE_CLASSES[this.props.size]] = true;
    } else {
      classes[BASE_SIZE_CLASSES[BaseSize.Default]] = true;
    }

    // Because Gatsby runs these files on both the server and in the browser,
    // we need a way to parse the DOM without using the window object (that
    // is not availabe on the server). To do this, we use the `xmldom`
    // package to provide a parser and serializer.
    const DOM = this.parseFromString(this.props.source);
    const nodes = Array.from(DOM.childNodes);

    const content: (JSX.Element | undefined)[] = nodes.map((node, index) => {
      if (!node.childNodes) {
        return;
      }

      const preBlock = findChild(node, /pre/);

      // YML blocks - our custom directives for convinience
      if (preBlock && this.getCodeBlockSettings(preBlock).match("yml")) {
        return this.renderCustomBlock(preBlock, index);
      }

      if (preBlock && this.getCodeBlockSettings(preBlock).match("code-.*")) {
        return this.renderSourceCodeText(preBlock, index);
      }

      // Code examples
      if (preBlock && this.getCodeBlockSettings(preBlock).match("jsx")) {
        return this.renderCodeBlock(preBlock, index);
      }

      if (node.nodeName.match(/h[1-6]/)) {
        return this.renderHeading(node, index);
      }

      // All other ordinary content
      return this.renderMarkdownBlock(node, index);
    });

    return (
      <div className={`${classnames("markdown", classes)}`}>{content}</div>
    );
  }

  private renderSourceCodeText = (elem: ChildNode, key: number) => {
    const settings = this.getCodeBlockSettings(elem);
    return (
      <Layout margin={{ y: 3 }} key={`markdown-${key}`}>
        <CodeEditor
          children={elem.textContent || ""}
          language={settings.replace(/\s?code-/, "")}
          truncate={true}
        />
      </Layout>
    );
  };

  private renderCodeBlock = (elem: ChildNode, key: number) => {
    return (
      <Layout margin={{ y: 3 }} key={`markdown-${key}`}>
        <CodePreview renderCode={elem.textContent || ""} />
      </Layout>
    );
  };

  private renderCustomBlock = (elem: ChildNode, key: number) => {
    const settings = this.getCodeBlockSettings(elem);
    const yamlString = elem.textContent
      ? elem.textContent.replace(/^\s+/, "")
      : "";
    const data = YAML.load(yamlString);

    return (
      <Layout margin={{ y: 3 }} key={key}>
        {settings.match(/block-links/) && <BlockLinks key={key} data={data} />}
        {settings.match(/block-columns/) && (
          <BlockColumns key={key} data={data} />
        )}
        {settings.match(/block-guideline/) && (
          <BlockGuidelines key={key} data={data} />
        )}
        {settings.match(/block-media/) && <BlockMedia key={key} data={data} />}
      </Layout>
    );
  };

  private renderHeading = (elem: ChildNode, index: number) => {
    let html = this.serializeToString(elem);
    const { nodeName } = elem;

    const sizeMap = {
      h1: TitleSize.ExtraLarge,
      h2: TitleSize.Large,
      h3: TitleSize.Default,
      h4: TitleSize.Small,
      h5: TitleSize.ExtraSmall,
      h6: TitleSize.ExtraSmall,
    };

    const marginMap: { [key: string]: Margin } = {
      h1: { top: index === 0 ? 0 : 3, bottom: 2 },
      h2: { top: index === 0 ? 0 : 3, bottom: 2 },
      h3: { top: index === 0 ? 0 : 2, bottom: 2 },
      h4: { top: index === 0 ? 0 : 2, bottom: 2 },
      h5: { top: index === 0 ? 0 : 2, bottom: 2 },
      h6: { top: index === 0 ? 0 : 2, bottom: 2 },
    };

    return (
      <Layout
        className="markdown__block"
        key={`markdown-${index}`}
        margin={
          nodeName in marginMap
            ? marginMap[nodeName as keyof typeof marginMap]
            : undefined
        }
      >
        <Title
          type={TextType.Span} // The actual 'h1' tag is already present in the child
          size={
            nodeName in sizeMap
              ? sizeMap[nodeName as keyof typeof sizeMap]
              : undefined
          }
        >
          {renderHTML(html)}
        </Title>
      </Layout>
    );
  };

  private renderMarkdownBlock = (elem: ChildNode, index: number) => {
    let html = this.serializeToString(elem);

    // If this element is a video embed iframe (from Google Drive, YouTube,
    // or Vimeo), wrap it in an Aspect component with a background.
    if (html.match(/\<iframe.*(drive\.google|youtube|vimeo).*\<\/iframe\>/)) {
      return (
        <StyledLayout
          background={Background.Alt}
          className="markdown__iframe"
          margin={{ y: 3 }}
        >
          <Aspect ratio={AspectRatio.Aspect16x9}>{renderHTML(html)}</Aspect>
        </StyledLayout>
      );
    }

    // Check for color variable references and append swatches next to them.
    const colorVarRefs =
      html.match(
        /\$(white|black|red|orange|yellow|green|blue|prime-blue|info|success|error|warn|opac-b|opac-w|twitch\-purple|hinted\-grey|color-(\w*))(\-[0-9]{1,2}){0,1}/g,
      ) || [];
    colorVarRefs.map(ref => {
      html = html.replace(
        `<code>${ref}</code>`,
        `
          <div className="tw-inline-flex tw-align-items-center">
            <code data-replaced="true">${ref}</code>
            <div class="markdown__color-swatch markdown__color-swatch--${ref.replace(
              "$",
              "",
            )} tw-mg-l-05"></div>
          </div>
        `,
      );
    });

    return (
      <Layout className="markdown__block" key={`markdown-${index}`}>
        {renderHTML(html)}
      </Layout>
    );
  };

  private getCodeBlockSettings = (elem: ChildNode) => {
    let attribute: Attr | undefined = Array.from(
      (elem as Element).attributes,
    ).find((attr: Attr) => attr.name === "data-language");

    if (!attribute && elem.firstChild) {
      attribute = Array.from((elem.firstChild as Element).attributes).find(
        (attr: Attr) => attr.name === "class",
      );
    }
    return attribute ? attribute.value.replace(/language\-/, "") : "";
  };

  private serializeToString = (elem: ChildNode) => {
    const serializer = new XMLSerializer();
    return serializer.serializeToString(elem);
  };

  private parseFromString = (source: string) => {
    const parser = new DOMParser();
    return parser.parseFromString(source, "text/html");
  };
}

function findChild(
  node: ChildNode,
  pattern: string | RegExp,
): ChildNode | undefined {
  if (node.nodeName && node.nodeName.match(pattern)) {
    return node;
  }

  if (!node.childNodes) {
    return undefined;
  }

  return Array.from(node.childNodes).find(
    n => !!(n.nodeName && n.nodeName.match(pattern)),
  );
}
