import { LocationContext } from "@reach/router";
import * as React from "react";
import {
  Column,
  Grid,
  JustifyContent,
  StyledLayout,
  TabItemPublicProps,
  Tabs,
  TabSize,
} from "twitch-core-ui";
import { File, FileConnection, FileEdge } from "../../graphql-types";
import { renderLink } from "../../utils/render-link";
import { sortByRelativePath } from "../../utils/sort-by-relative-path";
import "./styles.scss";

interface PublicProps {
  page: File;
  tabs: FileConnection;
  isComponent?: boolean;
}

type Props = PublicProps & LocationContext;

export class PageTabs extends React.Component<Props, {}> {
  constructor(props: Props) {
    super(props);
  }

  public render() {
    const tabs = this.getTabs();
    const activeTabIndex = this.props.location.pathname
      ? tabs.findIndex(item => {
          return (
            !!item.linkTo &&
            this.props.location.pathname
              .replace(/\/$/, "")
              .endsWith(item.linkTo.toString().replace(/\/$/, ""))
          );
        })
      : 0;

    return (
      <StyledLayout borderBottom className="page-tabs">
        <Grid>
          <Column cols={{ default: 6, md: 8 }}>
            {tabs.length > 1 && (
              <Tabs
                tabs={tabs}
                activeTabIndex={activeTabIndex}
                size={TabSize.Large}
                justifyContent={JustifyContent.Start}
                borderBottom={false}
              />
            )}
          </Column>
        </Grid>
      </StyledLayout>
    );
  }

  private getTabs = (): TabItemPublicProps[] => {
    const tabs: TabItemPublicProps[] = [];
    let { edges } = this.props.tabs;

    let indexTab =
      edges &&
      edges.find(
        edge =>
          (edge &&
            edge.node &&
            edge.node.fields &&
            edge.node.fields.tab &&
            edge.node.fields.tab.slug === "index") ||
          false,
      );

    if (
      indexTab &&
      indexTab.node &&
      indexTab.node.fields &&
      indexTab.node.fields.tabPath &&
      indexTab.node.fields.tab
    ) {
      tabs.push({
        label: this.getTitle(indexTab),
        linkTo: indexTab.node.fields.tabPath,
        renderLink,
      });
    }

    sortByRelativePath(edges || []).forEach(edge => {
      if (
        edge.node &&
        edge.node.fields &&
        edge.node.fields.tabPath &&
        edge.node.fields.tab &&
        edge.node.fields.tab.slug !== "index"
      ) {
        tabs.push({
          label: this.getTitle(edge),
          linkTo: edge.node.fields.tabPath,
          renderLink,
        });
      }
    });

    return tabs;
  };

  private getTitle = (edge: FileEdge) => {
    if (
      edge.node &&
      edge.node.childMarkdownRemark &&
      edge.node.childMarkdownRemark.frontmatter &&
      edge.node.childMarkdownRemark.frontmatter.title
    ) {
      return edge.node.childMarkdownRemark.frontmatter.title;
    }

    if (
      edge.node &&
      edge.node.fields &&
      edge.node.fields.tab &&
      edge.node.fields.tab.title
    ) {
      return edge.node.fields.tab.title;
    }

    return "";
  };
}
