import { LocationContext } from "@reach/router";
import { graphql } from "gatsby";
import * as React from "react";
import { Helmet } from "react-helmet";
import { Layout, ResponsiveWrapper } from "twitch-core-ui";
import { File, FileConnection, SitePage, TypeDoc } from "../../graphql-types";
import { PageHeading } from "../page-heading";
import { PageTabs } from "../page-tabs";

interface PublicProps {
  children?: React.ReactNode;
  page: SitePage;
  file: File;
  tabs: FileConnection;
  component?: TypeDoc;
}

type Props = PublicProps & LocationContext;

export const PageWrapper = (props: Props) => {
  return (
    <Layout padding={5}>
      {renderPageTitle(props.page)}
      <ResponsiveWrapper>
        <PageHeading page={props.file} />
        <Layout margin={{ top: 3, bottom: 4 }}>
          <PageTabs
            page={props.file}
            tabs={props.tabs}
            location={props.location}
            navigate={props.navigate}
          />
        </Layout>
        {props.children}
      </ResponsiveWrapper>
    </Layout>
  );
};

const renderPageTitle = (page: SitePage) => {
  if (page.context && page.context.title) {
    return (
      <Helmet>
        <title>{page.context.title}</title>
      </Helmet>
    );
  }
};

export const defaultPageQuery = graphql`
  fragment defaultPage on Query {
    indexPage: sitePage(path: { eq: $path }) {
      id
      context {
        title
      }
    }
    indexFile: file(
      fields: { path: { eq: $pagePath }, tab: { slug: { eq: "index" } } }
    ) {
      modifiedTime
      relativePath
      fields {
        title
        path
      }
      childMarkdownRemark {
        frontmatter {
          title
          description
        }
      }
    }
    tabs: allFile(filter: { fields: { path: { eq: $pagePath } } }) {
      edges {
        node {
          relativePath
          fields {
            tabPath
            tab {
              title
              slug
            }
          }
        }
      }
    }
  }
`;
