import { LocationContext } from "@reach/router";
import { debounce } from "lodash";
import * as queryString from "query-string";
import * as React from "react";
import {
  AlignItems,
  Background,
  BorderRadius,
  ButtonIcon,
  CoreText,
  Display,
  FlexDirection,
  FlexWrap,
  FontSize,
  JustifyContent,
  Layout,
  Overflow,
  Position,
  StyledLayout,
  SVGAsset,
  Tooltip,
  TooltipDirection,
} from "twitch-core-ui";
import { CodeEditor } from "../../components/code-editor";
import { CodePreview } from "../../components/code-preview";
import { Overlay } from "../../components/overlay";
import { TypeDoc } from "../../graphql-types";
import {
  ComponentPlaygroundProperty,
  PropField,
} from "./components/prop-field";
import {
  propsToRenderString,
  setupDefaultPropValues,
} from "./utils/property-helpers";

import "./styles.scss";

interface PublicProps {
  file: TypeDoc;
  onClickClose?: () => void;
}

interface State {
  props: ComponentPlaygroundProperty[];
  query?: { [key: string]: string };
  previewWidth?: number;
  previewShouldCalculate?: boolean;
}

type Props = PublicProps & LocationContext;

export class Playground extends React.Component<Props, State> {
  private previewWrapperElement: HTMLDivElement | null;
  private previewElement: HTMLDivElement | null;

  constructor(props: Props) {
    super(props);

    this.state = {
      props: setupDefaultPropValues(this.props.file.properties || []),
      previewShouldCalculate: true,
    };

    this.updateURLParams = debounce(this.updateURLParams, 500);
  }

  public componentDidMount() {
    this.updatePropsFromURL();
    this.calculateAndSetPreviewSize();

    // We have to wait until everything has loaded.
    window.addEventListener("load", this.calculateAndSetPreviewSize);
    window.addEventListener("resize", this.calculateAndSetPreviewSize);
  }

  public componentWillUnmount() {
    window.removeEventListener("load", this.calculateAndSetPreviewSize);
    window.removeEventListener("resize", this.calculateAndSetPreviewSize);
  }

  public render() {
    let showOverlay: boolean =
      undefined !==
      this.state.props.find(
        prop =>
          (prop.currentValue &&
            prop.type === "boolean" &&
            prop.name === "overlay") ||
          false,
      );

    const requiredPropertyList = this.state.props
      .filter(prop => prop.flags === null || !prop.flags.isOptional)
      .map((props, index) => (
        <PropField onValueChange={this.updatePreview} key={index} {...props} />
      ));

    const optionalPropertyList = this.state.props
      .filter(prop => prop.flags !== null && prop.flags.isOptional)
      .map((props, index) => (
        <PropField onValueChange={this.updatePreview} key={index} {...props} />
      ));

    return (
      <Overlay>
        <StyledLayout
          className="playground"
          borderRadius={BorderRadius.Medium}
          overflow={Overflow.Hidden}
          fullHeight
          fullWidth
        >
          <StyledLayout
            background={Background.Base}
            position={Position.Absolute}
            fullHeight
            attachTop
            attachLeft
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            flexWrap={FlexWrap.NoWrap}
            flexShrink={0}
            className="playground__sidebar"
          >
            <StyledLayout
              className="playground__header"
              display={Display.Flex}
              alignItems={AlignItems.Center}
              flexShrink={0}
              padding={{ x: 2, y: 1 }}
            >
              <Layout flexGrow={1}>
                <CoreText fontSize={FontSize.Size4} bold>
                  {this.props.file.name}
                </CoreText>
              </Layout>
              <Tooltip
                label="Reset Properties"
                direction={TooltipDirection.Bottom}
              >
                <ButtonIcon
                  icon={SVGAsset.VideoRerun}
                  ariaLabel={`Reset`}
                  onClick={this.resetProps}
                />
              </Tooltip>
            </StyledLayout>
            <Layout overflow={Overflow.Auto} padding={{ y: 2 }}>
              {requiredPropertyList}
              {optionalPropertyList}
            </Layout>
          </StyledLayout>
          <StyledLayout
            background={Background.Alt}
            display={Display.Flex}
            flexDirection={FlexDirection.Column}
            fullHeight
            className="playground__main"
          >
            <StyledLayout
              className="playground__header"
              display={Display.Flex}
              alignItems={AlignItems.Center}
              flexShrink={0}
            >
              <Layout flexGrow={1} padding={1}>
                {/* Add a quick share link. */}
              </Layout>
              <StyledLayout padding={1} borderLeft>
                <ButtonIcon
                  onClick={this.props.onClickClose}
                  icon={SVGAsset.Close}
                  ariaLabel={`Back to ${this.props.file.name}`}
                />
              </StyledLayout>
            </StyledLayout>
            <Layout
              padding={4}
              display={Display.Flex}
              position={Position.Relative}
              flexGrow={2}
              alignItems={AlignItems.Center}
              justifyContent={JustifyContent.Center}
              className="playground__preview"
            >
              <Layout
                refDelegate={ref => (this.previewWrapperElement = ref)}
                fullWidth
                fullHeight
              >
                <CodePreview
                  calculateWidth={this.state.previewShouldCalculate}
                  width={this.state.previewWidth}
                  refDelegate={ref => (this.previewElement = ref)}
                  renderCode={propsToRenderString(
                    this.props.file.name || "NULL",
                    this.state.props,
                    false,
                  )}
                  showOverlay={showOverlay}
                  centerContent
                />
              </Layout>
            </Layout>
            <Layout
              flexGrow={1}
              padding={{ bottom: 3, x: 4 }}
              className="playground__editor"
            >
              <CodeEditor language="jsx">
                {propsToRenderString(
                  this.props.file.name || "NULL",
                  this.state.props,
                  true,
                )}
              </CodeEditor>
            </Layout>
          </StyledLayout>
        </StyledLayout>
      </Overlay>
    );
  }

  /**
   * Update the playground preview and code based on input.
   */
  private updatePreview = (name: string, value: string | boolean | null) => {
    let query = this.state.query;
    let props = this.state.props;

    let updatedProp = props.find(p => p.name === name);

    if (updatedProp) {
      updatedProp.currentValue = value;

      if (updatedProp.currentValue && updatedProp.name && query) {
        query[updatedProp.name] = updatedProp.currentValue.toString();
      } else if (updatedProp.name && query) {
        delete query[updatedProp.name];
      }
    }

    this.updateStateAndQuery(props, query);

    // Wait until the next frame before calculating the new preview size.
    window.requestAnimationFrame(this.calculateAndSetPreviewSize);
  };

  /**
   * This method gets the aspect ratio of the contents of the preview and
   * sets a width to ensure they will always fit vertically in the area.
   */
  private calculateAndSetPreviewSize = () => {
    this.setState({ previewWidth: undefined, previewShouldCalculate: true });

    if (!this.previewWrapperElement || !this.previewElement) {
      return;
    }

    if (
      this.previewWrapperElement.clientHeight > this.previewElement.clientHeight
    ) {
      this.setState({ previewShouldCalculate: false });
      return;
    }

    const previewAspectRatio =
      this.previewElement.clientHeight / this.previewElement.clientWidth;
    this.setState({
      previewWidth:
        this.previewWrapperElement.clientHeight / previewAspectRatio,
    });
  };

  /**
   * Reset the properties to the original values.
   */
  private resetProps = () => {
    let props = this.state.props;
    let query = this.state.query;

    props.map(prop => {
      // Return the current value to the original value.
      prop.currentValue = prop.originalValue ? prop.originalValue : "";

      // Remove the query paramater from the URL.
      if (prop.name && query) {
        delete query[prop.name];
      }
    });

    this.updateStateAndQuery(props, query);
  };

  /**
   * A method to update both the state and URL query – they need to always
   * be mirrored.
   */
  private updateStateAndQuery = (props: any, query: any) => {
    this.updateURLParams(query);
    this.setState({ props, query });
  };

  private updatePropsFromURL = () => {
    let query = queryString.parse(this.props.location.search);
    let propertiesWithValues = setupDefaultPropValues(
      this.props.file.properties || [],
    );

    // Set up the playground based on the query string:
    // Match keys and values from the query string.
    Object.keys(query).map(key => {
      let prop = propertiesWithValues.find(p => p.name === key);

      if (prop) {
        prop.currentValue = query[key];
      }
    });

    this.setState({
      props: propertiesWithValues,
      query,
    });
  };

  private updateURLParams = (query: any) => {
    this.props.navigate(
      `${this.props.location.pathname}${
        query ? "?" + queryString.stringify(query) : ""
      }`,
    );
  };
}
