import * as React from "react";
import {
  AlignItems,
  CoreText,
  Display,
  FontSize,
  Input,
  InputType,
  Layout,
  Select,
  Toggle,
} from "twitch-core-ui";
import { properties_3 as TypeDocProperty } from "../../../../graphql-types";
import { Code } from "../../../code";
import { formatAvailableValue } from "../../utils/property-helpers";
import "./styles.scss";

export interface ComponentPlaygroundProperty extends TypeDocProperty {
  currentValue: string | boolean | null;
  originalValue: string | boolean | null;
}

export interface Props extends ComponentPlaygroundProperty {
  onValueChange: (a: string | null, b: string | boolean | null) => void;
}

export class PropField extends React.Component<Props> {
  public render() {
    const isRequired =
      this.props.flags === null || !this.props.flags.isOptional;
    return (
      <Layout
        className="prop-field"
        display={Display.Flex}
        alignItems={AlignItems.Center}
        key={this.props.name || ""}
        padding={{ x: 2, y: 1 }}
      >
        <Layout flexGrow={1}>
          <Layout padding={{ bottom: 0.5 }}>
            <CoreText fontSize={FontSize.Size6} bold>
              <Code required={isRequired}>
                {this.props.name}
                {!isRequired && "?"}
              </Code>
            </CoreText>
          </Layout>
          {this.props.type && (
            <CoreText fontSize={FontSize.Size7}>
              <Code>{this.props.type}</Code>
            </CoreText>
          )}
        </Layout>
        <Layout flexGrow={1} className="prop-field__input-wrapper">
          {this.renderFormField()}
        </Layout>
      </Layout>
    );
  }

  private renderFormField() {
    let inputCurrentValue = this.props.currentValue
      ? this.props.currentValue.toString()
      : "";

    if (
      this.props.type &&
      this.props.type.match(/React\.React(Node|Element)/)
    ) {
      return (
        <Input
          type={InputType.Text}
          name={this.props.name || ""}
          value={inputCurrentValue}
          onChange={this.handleInput}
        />
      );
    }

    if (this.props.type === "string") {
      return (
        <Input
          type={InputType.Text}
          name={this.props.name || ""}
          value={inputCurrentValue}
          onChange={this.handleInput}
        />
      );
    }

    if (this.props.type === "number") {
      return (
        <Input
          type={InputType.Number}
          name={this.props.name || ""}
          value={inputCurrentValue}
          onChange={this.handleInput}
        />
      );
    }

    if (this.props.type === "boolean") {
      return (
        <Toggle
          checked={!!inputCurrentValue}
          name={this.props.name || ""}
          onChange={this.handleInput}
        />
      );
    }

    if (this.props.type === null) {
      return undefined;
    }

    if (this.props.type && this.props.type.match(/BreakpointProps/)) {
      return undefined;
    }

    if (this.props.type && this.props.type.match(/^React\.(.*)EventHandler$/)) {
      return undefined;
    }

    if (this.props.availableValues && this.props.availableValues !== null) {
      return (
        <Select
          onChange={this.handleSelect}
          name={this.props.name || ""}
          value={inputCurrentValue}
        >
          {this.props.flags && this.props.flags.isOptional ? <option /> : ""}
          {this.props.availableValues.map(
            entry =>
              entry.values &&
              entry.values.map(value => (
                <option value={formatAvailableValue(value, entry)} key={value}>
                  {value}
                </option>
              )),
          )}
        </Select>
      );
    }

    return undefined;
  }

  private handleSelect = (e: React.ChangeEvent<HTMLSelectElement>) => {
    let value: string | null =
      e.target.value.length > 0 ? e.target.value : null;
    this.props.onValueChange(e.target.name, value);
  };

  private handleInput = (e: React.ChangeEvent<HTMLInputElement>) => {
    let value: string | boolean = e.currentTarget.value;

    // If the prop field renders a Toggle input, use the `checked` key for
    // the value.
    if (e.currentTarget.type === "checkbox") {
      value = e.currentTarget.checked;
    }

    this.props.onValueChange(this.props.name, value);
  };
}
