import {
  availableValues_2 as AvailableValues,
  properties_3 as TypeDocProperty,
} from "../../../graphql-types";
import { ComponentPlaygroundProperty } from "../components/prop-field";

export const CHILDREN_REGEX_MATCHER = /React\.React(Node|Element)/;
export const ENUM_TYPE_STRING = "Enumeration";

/**
 * Takes TypeDocProps and populates some default values
 */
export function setupDefaultPropValues(
  props: TypeDocProperty[],
): ComponentPlaygroundProperty[] {
  return props.map(prop => {
    let originalValue = "";
    let currentValue = "";

    // Set up the default values:
    //
    // 1. If the property is required and has available values, make the first
    //    available value the currently selected value.
    if (prop.flags && !prop.flags.isOptional && prop.availableValues) {
      // Get the first element in the nested array... :p
      const value =
        prop.availableValues.reduce((a1, item) => {
          const defaultValue =
            a1 || (item.values && item.values.reduce((a2, v) => a2 || v));
          if (defaultValue) {
            return formatAvailableValue(defaultValue, item);
          }
        }, "") || "";

      originalValue = value;
      currentValue = value;
    }

    // 2. If an example value has been given via a TypeDoc tag, set that as
    //    the currently selected value.
    if (prop.comment && prop.comment.tags) {
      const tag = prop.comment.tags.find((t: any) => t.tag === "example");
      let value = tag && tag.text ? tag.text.trim() : "";

      // Ensure that example enum values passed in via TypeDoc comments are
      // formatted correctly.
      const availableValues =
        prop.availableValues &&
        prop.availableValues.find(
          valueSet => valueSet.type === ENUM_TYPE_STRING,
        );

      if (availableValues) {
        value = formatAvailableValue(value, availableValues);
      }

      originalValue = value;
      currentValue = value;
    }

    const data: ComponentPlaygroundProperty = {
      ...prop,
      originalValue,
      currentValue,
    };

    return data;
  });
}

/**
 * Takes an array of component props, and turns it into a string of the JSX code to render the component
 */
export function propsToRenderString(
  componentName: string,
  props: ComponentPlaygroundProperty[],
  humanReadable: boolean = false,
): string {
  const propSeparator = humanReadable ? "\n\xa0\xa0" : " ";
  const br = humanReadable ? "\n" : "";

  let renderCode: string = "";
  let propString: string = "";
  let children: string | undefined;

  // Map over the select props array to construct the property string and children.
  props.map(prop => {
    // Build the property string. This will be used to display in the code
    // snippet and render the preview.

    if (!prop.currentValue) {
      return;
    }

    let currentValue = prop.currentValue.toString().trim();

    if (prop.type === "string") {
      propString += `${propSeparator}${prop.name}="${currentValue}"`;
    }

    if (prop.type === "number") {
      propString += `${propSeparator}${prop.name}={${currentValue}}`;
    }

    if (prop.type === "boolean") {
      propString += `${propSeparator}${prop.name}`;
    }

    if (prop.availableValues && prop.availableValues.length > 0) {
      propString += `${propSeparator}${prop.name}={${prop.currentValue}}`;
    }

    if (prop.type && prop.type.match(CHILDREN_REGEX_MATCHER)) {
      children = `${currentValue}`;
    }
  });

  // If there are no selected props, the component Tag should be on a single line.
  if (propString.length > 0) {
    propString += br;
  }

  // Build the full code snippet.
  // Note: The structuring of these strings is important for Gatsby to build
  //       correctly. If there is an empty space inbetween some brackets,
  //       React will throw an error on build.
  if (children) {
    if (humanReadable) {
      children = `${propSeparator}${children}${br}`;
    }

    // If the children prop is applied, we want an opening and closing tag.
    renderCode = `<${componentName}${propString}>${children}</${componentName}>`;

    // Hard-coded logic for better rendering:
    //
    // 1. If this is a balloon, add a relatively positioned container.
    if (componentName && componentName.match(/Balloon/)) {
      renderCode = humanReadable ? indent(renderCode, 2) : renderCode;
      renderCode = `<Layout position={Position.Relative}>${br}${renderCode}${br}</Layout>`;
    }
  } else {
    // If there are no children, snippet should display a self closing tag.
    renderCode = `<${componentName} ${
      propString.length > 0 ? propString : " "
    }/>`;
  }

  return renderCode;
}

/**
 * Determines how to format an available value into a string with the
 * correct syntax.
 */
export function formatAvailableValue(
  value: string,
  availableValues: AvailableValues,
) {
  return `${
    availableValues.type &&
    availableValues.type === "Enumeration" &&
    availableValues.name &&
    value.indexOf(availableValues.name) === -1
      ? availableValues.name + "."
      : ""
  }${value}`;
}

function indent(str: string, spaces: number) {
  return str.replace(/^(?=.)/gm, new Array(spaces + 1).join("\xa0"));
}
