import { LocationContext } from "@reach/router";
import { navigate } from "gatsby";
import { debounce } from "lodash";
import * as queryString from "query-string";
import * as React from "react";
import {
  Background,
  BorderRadius,
  InjectLayout,
  Overflow,
  Position,
  StyledLayout,
  ZIndex,
} from "twitch-core-ui";
import { NoResults } from "./../no-results";
import { SiteSearchInput } from "./../search-input";
import { SearchResult } from "./../search-result";
import { FuseMatch, search, SearchItem } from "./../utils/search-utils";
import "./styles.scss";

const TRACKING_DEBOUNCE_TIMEOUT = 250;
const NUM_RESULTS = 8;
const ACTIVE_SEARCH_WIDTH = "28rem";

export interface PublicProps {
  items: SearchItem[];
  onChange?: (searchTerm: string) => void;
}

type Props = PublicProps & LocationContext;

export interface State {
  focusIndex: number;
  searchTerm: string;
  matches: FuseMatch[];
  searchIsFocused: boolean;
}

export class Search extends React.Component<Props, State> {
  private inputElement: HTMLElement | null;

  constructor(props: Props) {
    super(props);

    const query = queryString.parse(props.location.search);
    const searchTerm = (query && query.search) || "";

    if (this.props.onChange) {
      this.props.onChange(searchTerm);
    }

    this.state = {
      focusIndex: -1,
      matches: this.getMatches(searchTerm),
      searchTerm: (query && query.search) || "",
      searchIsFocused: false,
    };
  }

  public render() {
    return (
      <InjectLayout
        className="search"
        position={Position.Relative}
        padding={0.5}
      >
        <div
          onKeyDown={this.onSearchInputKeyDown}
          style={{
            width: this.state.searchIsFocused ? ACTIVE_SEARCH_WIDTH : undefined,
          }}
        >
          <SiteSearchInput
            onFocus={this.onSearchInputFocus}
            onBlur={this.onSearchInputBlur}
            onChange={this.onSearchInputChange}
            term={this.state.searchTerm}
            refDelegate={ref => (this.inputElement = ref)}
          />
          {this.state.searchIsFocused && this.state.searchTerm.length > 0 && (
            <StyledLayout
              className={`search__results`}
              overflow={Overflow.Auto}
              background={Background.Base}
              borderRadius={BorderRadius.Medium}
              padding={{ x: 0.5, top: 4, bottom: 0.5 }}
              elevation={3}
              position={Position.Absolute}
              zIndex={ZIndex.Below}
              attachTop
              attachLeft
              fullWidth
            >
              {this.state.matches.length > 0 &&
                this.state.matches.map((item, index) => (
                  <SearchResult
                    onClick={this.selectSearchResult}
                    isFocused={this.state.focusIndex === index}
                    item={item}
                    index={index}
                    key={index}
                    onMouseEnter={this.onSearchResultMouseEnter}
                  />
                ))}
              {this.state.matches.length === 0 && <NoResults />}
            </StyledLayout>
          )}
        </div>
      </InjectLayout>
    );
  }

  private onSearchInputFocus = () => this.setState({ searchIsFocused: true });
  private onSearchInputBlur = () => {
    setTimeout(() => {
      this.setState({ searchIsFocused: false });
      this.clearSearch();
    }, 150);
  };

  private onSearchInputChange = (e: React.KeyboardEvent<HTMLInputElement>) => {
    const searchTerm = e.currentTarget.value;

    // Set the search term.
    this.setSearchTerm(searchTerm);

    this.setState({
      focusIndex: 0,
      searchIsFocused: true, // it's possible to type in the box without triggering onFocus
      matches: this.getMatches(searchTerm),
    });
  };

  private getMatches = (searchTerm: string) => {
    return searchTerm.length > 0
      ? search(searchTerm, this.props.items).slice(0, NUM_RESULTS)
      : ([] as FuseMatch[]);
  };

  private onSearchInputKeyDown = (e: React.KeyboardEvent<HTMLDivElement>) => {
    if (
      e.key === "ArrowDown" &&
      this.state.focusIndex < this.state.matches.length - 1
    ) {
      this.setState({ focusIndex: this.state.focusIndex + 1 });
    }

    if (e.key === "ArrowUp" && this.state.focusIndex > 0) {
      this.setState({ focusIndex: this.state.focusIndex - 1 });
    }

    if (e.key === "Enter" && this.state.focusIndex > -1) {
      // Find the search result.
      this.selectSearchResult(this.state.focusIndex);
    }

    if (e.key === "Escape") {
      if (this.state.searchTerm.length > 0) {
        this.clearSearch();
      } else {
        this.blurSearchInput();
      }
    }
  };

  private blurSearchInput = () => {
    if (this.inputElement) {
      this.inputElement.blur();
    }
  };

  private onSearchResultMouseEnter = (index: number) => {
    this.setState({ focusIndex: index });
  };

  private selectSearchResult = (index: number) => {
    // this.setState({ searchIsFocused: true });

    const currentMatch = this.state.matches[index];

    // Send before we navigate to a result page.
    this.debounceTrackSearchTerm.flush();

    // Navigate to the page.
    navigate(currentMatch.item.path);

    this.blurSearchInput();
  };

  private clearSearch = () => {
    // Reset the search term.
    this.setSearchTerm("");

    // Clear the matches array.
    this.setState({ matches: [] });
  };

  private setSearchTerm = (searchTerm: string) => {
    this.setState({
      searchTerm,
    });

    this.debounceTrackSearchTerm();

    if (this.props.onChange) {
      this.props.onChange(searchTerm);
    }
  };

  // tslint:disable-next-line:member-ordering
  private debounceTrackSearchTerm = debounce(() => {
    // Don't replace history state if there is no change
    const currentQuery = queryString.parse(window.location.search);
    if (currentQuery.search === undefined && this.state.searchTerm === "") {
      return;
    }

    // Do use an empty string if there is no search term
    let term = "";

    if (this.state.searchTerm) {
      term = queryString.stringify({
        search: this.state.searchTerm,
      });
    }

    this.props.navigate(
      `${this.props.location.pathname}${term ? "?" + term : ""}`,
      { replace: true },
    );
  }, TRACKING_DEBOUNCE_TIMEOUT);
}
