import * as Fuse from "fuse.js";
import { graphql } from "gatsby";
import {
  File,
  FileConnection,
  SitePage,
  SitePageConnection,
} from "src/graphql-types";

export interface SearchItem {
  path: string;
  title: string;
  headings?: any;
  body?: string;
  properties?: any;
  excerpt?: string;
}

export interface FuseMatch {
  item: SearchItem;
  score: number;
  matches: any;
}

const fuseConfig = {
  caseSensitive: false,
  shouldSort: true,
  includeScore: true,
  minMatchCharLength: 3,
  includeMatches: true,
  threshold: 0.5,
  findAllMatches: true,
  keys: [
    // Note: Sum of weights should be between 0 and 1;
    // Avoid setting weight of "1" because it causes weird stuff
    { name: "title", weight: 0.45 },
    { name: "path", weight: 0.25 },
    { name: "headings", weight: 0.05 },
    { name: "body", weight: 0.05 },
    { name: "properties.name", weight: 0.01 },
    { name: "properties.type", weight: 0.01 },
    { name: "properties.comment", weight: 0.005 },
  ],
};

export function search(term: string, items: SearchItem[]): FuseMatch[] {
  const myFuse = new Fuse(items, fuseConfig);
  // @ts-ignore
  return myFuse.search(term);
}

export function dataToSearchObjects(
  allPages: SitePageConnection,
  allFiles: FileConnection,
): SearchItem[] {
  if (!allPages.edges) {
    return [];
  }

  return allPages.edges
    .filter(edge => {
      const page = edge.node;
      const file = page && findFile(page, allFiles);

      if (page === null) {
        return false;
      }

      if (page.path && page.path === "/dev-404-page/") {
        return false;
      }

      // Don't show these tabs for now; makes search results too messy. We will implement custom result cards for these
      if (
        page.path &&
        (page.path.endsWith("properties") ||
          page.path.endsWith("changelog") ||
          page.path.endsWith("playground"))
      ) {
        return false;
      }

      // Don't show any results with a redirect frontmatter set
      if (
        file &&
        file.childMarkdownRemark &&
        file.childMarkdownRemark.frontmatter &&
        file.childMarkdownRemark.frontmatter.redirect
      ) {
        return false;
      }

      return true;
    })
    .map(edge => {
      const page = edge.node;
      const file = page && findFile(page, allFiles);

      return {
        path: (page && page.path) || "",
        title: (page && page.context && page.context.title) || "",
        excerpt:
          (file &&
            file.childMarkdownRemark &&
            file.childMarkdownRemark.frontmatter &&
            file.childMarkdownRemark.frontmatter.description) ||
          "",
        body:
          (file &&
            file.childMarkdownRemark &&
            file.childMarkdownRemark.htmlAst &&
            htmlAstToString(file.childMarkdownRemark.htmlAst)) ||
          "",
        headings:
          (file &&
            file.childMarkdownRemark &&
            file.childMarkdownRemark.headings &&
            file.childMarkdownRemark.headings.map(heading => heading.value)) ||
          "",
      };
    });
}

function findFile(page: SitePage, allFiles: FileConnection): File | undefined {
  const file =
    allFiles.edges &&
    allFiles.edges.find(item => {
      const filePath = item.node && item.node.fields && item.node.fields.path;
      const pagePath = page.path;
      return filePath === pagePath;
    });

  if (file && file.node) {
    return file.node;
  }
}

function htmlAstToString(htmlAst: any): string {
  // All text nodes will get returned!
  if (htmlAst.type === "text") {
    return (htmlAst.value || "").trim() + " ";
  }

  // If reaching a <pre> tag, stop and go no futher (i.e. markdown code examples)
  if (htmlAst.tagName === "pre") {
    return "";
  }

  // If node contains children, recursively search them all!
  if (htmlAst.children && htmlAst.children.length > 0) {
    return htmlAst.children
      .map((child: any) => {
        return htmlAstToString(child);
      })
      .join("");
  }

  // All other nodes
  return "";
}

// Fragment will be made available by Gatsby even though IDE will show zero usages of this exported variable
export const searchFragments = {
  pages: graphql`
    fragment SearchPagesFragment on SitePageConnection {
      edges {
        node {
          id
          path
          context {
            pagePath
            tab
            title
          }
        }
      }
    }
  `,
};

// Fragment will be made available by Gatsby even though IDE will show zero usages of this exported variable
export const searchFilesFragment = graphql`
  fragment SearchFilesFragment on FileConnection {
    edges {
      node {
        id
        relativePath
        fields {
          title
          path
        }
        childMarkdownRemark {
          id
          htmlAst
          headings {
            value
            depth
          }
          frontmatter {
            title
            description
            redirect
          }
        }
      }
    }
  }
`;
