import { LocationContext } from "@reach/router";
import { graphql, Link, StaticQuery } from "gatsby";
import { startCase } from "lodash";
import * as React from "react";
import {
  AlignItems,
  Background,
  Display,
  FlexDirection,
  FontSize,
  Icon,
  Layout,
  Overflow,
  StyledLayout,
  SVGAsset,
  VerticalNavigation,
  VerticalNavigationGroup,
  VerticalNavigationItem,
  VerticalNavigationStateProvider,
  VerticalNavigationSubHeader,
  VerticalNavigationTitle,
} from "twitch-core-ui";
import { FileConnection } from "../../graphql-types";
import { renderLink } from "../../utils/render-link";
import "./styles.scss";
import { getMenuData } from "./utils/get-menu-data";
import {
  getGroupPathname,
  getPathnameSegments,
  isCurrentPath,
} from "./utils/path-helpers";
interface GraphqlProps {
  data: {
    pages: FileConnection;
  };
}

type MenuListSubHeader = string;
interface MenuListItem {
  linkTo: string;
  label: string;
  subsectionIndex: number;
  subsectionPageIndex?: number;
}

interface MenuListGroup extends MenuListItem {
  items: MenuListItem[];
}

export type MenuDataList = Array<
  MenuListSubHeader | MenuListItem | MenuListGroup
>;

export interface SidebarMenuPublicProps extends LocationContext {}
export type SidebarMenuProps = GraphqlProps & SidebarMenuPublicProps;

export class SidebarMenuComponent extends React.Component<SidebarMenuProps> {
  public render() {
    const currentSection = getPathnameSegments(this.props.location.pathname)[0];
    const sectionTitle = startCase(currentSection);
    return (
      <StyledLayout
        className="sidebar-menu"
        display={Display.Flex}
        fontSize={FontSize.Size5}
        background={Background.Alt}
        fullHeight
      >
        <Layout
          display={Display.Flex}
          flexDirection={FlexDirection.Column}
          fullWidth
          fullHeight
        >
          {/* Home link */}
          <Layout
            margin={{ left: 2, top: 2 }}
            display={Display.Flex}
            alignItems={AlignItems.Center}
          >
            <Link to={"/"} style={{ textDecoration: "none" }}>
              <Layout display={Display.Flex} alignItems={AlignItems.Center}>
                <Icon asset={SVGAsset.ArrowLeft} />
                <Layout margin={{ left: 0.5 }}>Home</Layout>
              </Layout>
            </Link>
          </Layout>

          {/* Table of links */}
          <Layout overflow={Overflow.Auto}>
            <VerticalNavigationStateProvider
              defaultOpenGroupIDs={[
                getGroupPathname(this.props.location.pathname),
              ]}
            >
              {({ openGroupIDs, onOpenGroup, onCloseGroup }) => (
                <VerticalNavigation>
                  <VerticalNavigationTitle>
                    {sectionTitle}
                  </VerticalNavigationTitle>
                  <Layout padding={{ bottom: 1 }} />
                  {this.renderMenuItems(
                    openGroupIDs,
                    onOpenGroup,
                    onCloseGroup,
                  )}
                </VerticalNavigation>
              )}
            </VerticalNavigationStateProvider>
          </Layout>
        </Layout>
      </StyledLayout>
    );
  }

  private renderMenuItems = (
    openGroupIDs: string[],
    onOpenGroup: (id: string) => void,
    onCloseGroup: (id: string) => void,
  ) => {
    let menuItems: JSX.Element[] = [];

    // The `currentSection` refers to the highest-level docs sections. For
    // example, `user-interface` or `brand`.
    const currentSection = getPathnameSegments(this.props.location.pathname)[0];

    const menuData = getMenuData(this.props.data.pages, currentSection);

    // The vertical navigation refactor will have 4 main components:
    menuData.forEach((item, index) => {
      // VerticalNavigationSubHeader
      if (typeof item === "string") {
        menuItems.push(
          <React.Fragment key={`${currentSection}-menu-sub-header-${index}`}>
            <VerticalNavigationSubHeader>{item}</VerticalNavigationSubHeader>
          </React.Fragment>,
        );
      }

      // VerticalNavigationGroup
      if (isMenuGroup(item)) {
        menuItems.push(
          <VerticalNavigationGroup
            key={`${currentSection}-menu-group-${index}`}
            label={item.label}
            onOpen={() => onOpenGroup(item.linkTo)}
            onClose={() => onCloseGroup(item.linkTo)}
            open={openGroupIDs.includes(item.linkTo)}
          >
            {item.items.map((child, childIndex) =>
              this.renderVerticalNavigationItem(
                child,
                `menu-item-${index}-${childIndex}`,
              ),
            )}
          </VerticalNavigationGroup>,
        );
      }

      // VerticalNavigationItem
      if (isMenuItem(item)) {
        menuItems.push(
          this.renderVerticalNavigationItem(
            item,
            `${currentSection}-menu-item-${index}`,
          ),
        );
      }
    });

    return menuItems;
  };

  private renderVerticalNavigationItem = (item: MenuListItem, key: string) => (
    <VerticalNavigationItem
      key={key}
      renderLink={renderLink}
      linkTo={item.linkTo || ""}
      selected={
        isCurrentPath(this.props.location.pathname, item.linkTo) || undefined
      }
      blurAfterClick
    >
      {item.label}
    </VerticalNavigationItem>
  );
}

export const SidebarMenu = (props: SidebarMenuPublicProps) => (
  <StaticQuery
    render={data => <SidebarMenuComponent data={data} {...props} />}
    query={graphql`
      query {
        pages: allFile(
          sort: { fields: [relativePath], order: ASC }
          filter: {
            absolutePath: { regex: "/data/" }
            extension: { regex: "/md/" }
            fields: { tab: { slug: { eq: "index" } } }
          }
        ) {
          edges {
            node {
              relativePath
              childMarkdownRemark {
                frontmatter {
                  title
                }
              }
              fields {
                title
                path
                slug
                segments {
                  slug
                }
              }
            }
          }
        }
      }
    `}
  />
);

function isMenuGroup(item: any): item is MenuListGroup {
  return item.items !== undefined;
}

function isMenuItem(item: any): item is MenuListItem {
  return item.items === undefined && item.linkTo !== undefined;
}
